/***********************************************************
 * $Id$
 * 
 * Copyright (C) 2002-2006 ev-i Informationstechnologie GmbH
 *
 * Created: 03.09.2022
 *
 * Author: wpausch/ev-i
 * 
 ***********************************************************/

package at.cdes.api.document.compositeDto;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import at.cdes.api.joinDto.IDocumentListJoin;

/** Contains mappings for for all BASE components of T.
 *  E.g. documentIdToVersionJoins, documentVersionIdToCellResultJoins, and so on.
 */
public class DocumentListBaseContext<T extends IDocumentListJoin> {
    
    private Map<Long, List<T>> documentIdToJoins = new HashMap<Long, List<T>>();
    private Map<Long, List<T>> documentVersionIdToJoins = new HashMap<Long, List<T>>();
    private Map<Long, List<T>> cellResultIdToJoins = new HashMap<Long, List<T>>();
    private Map<Long, List<T>> nodeResultIdToJoins = new HashMap<Long, List<T>>();
    
    public DocumentListBaseContext(List<T> Ts) {
        for (T T : Ts) {
            Long documentId = T.getDocumentId();
            Long documentVersionId = T.getDocumentVersionId();
            Long cellResultId = T.getReviewCycleCellResultId();
            Long nodeResultId = T.getReviewCycleNodeResultId();

            if (documentId != null) {
                if (!this.documentIdToJoins.containsKey(documentId)) {
                    this.documentIdToJoins.put(documentId, new ArrayList<T>());
                }
                this.documentIdToJoins.get(documentId).add(T);                
            }
            if (documentVersionId != null) {
                if (!this.documentVersionIdToJoins.containsKey(documentVersionId)) {
                    this.documentVersionIdToJoins.put(documentVersionId, new ArrayList<T>());
                }
                this.documentVersionIdToJoins.get(documentVersionId).add(T);                
            }
            if (cellResultId != null) {
                if (!this.cellResultIdToJoins.containsKey(cellResultId)) {
                    this.cellResultIdToJoins.put(cellResultId, new ArrayList<T>());
                }
                this.cellResultIdToJoins.get(cellResultId).add(T);                
            }
            if (nodeResultId != null) {
                if (!this.nodeResultIdToJoins.containsKey(nodeResultId)) {
                    this.nodeResultIdToJoins.put(nodeResultId, new ArrayList<T>());
                }
                this.nodeResultIdToJoins.get(nodeResultId).add(T);                
            }
        }
    }
    
    // ========================== Document =========================
    
    public Set<Long> getAllDocumentIds() {
        return this.documentIdToJoins.keySet();
    }
    
    public boolean hasAnyDocumentVersion(Long documentId) {
        if (this.documentIdToJoins.containsKey(documentId)) {
            List<T> joins = this.documentIdToJoins.get(documentId);
            return !(joins.size() == 1 && joins.get(0).getDocumentVersionId() == null);
        } else {
            return false;
        }
    }
    
    public T getSomeJoinForDocumentId(Long documentId) {
        if (this.documentIdToJoins.containsKey(documentId)) {
            List<T> joins = this.documentIdToJoins.get(documentId);
            return joins.get(0);
        } else {
            return null;
        }
    }
    
    public T getMaxUploadedJoinForDocumentId(Long documentId) {
        List<T> currJoins = this.documentIdToJoins.get(documentId);
        T maxUploadedJoin = null;        
        if (currJoins != null) {
            for (T currJoin : currJoins) {
                Double currUploaded = currJoin.getDocumentVersionUploaded();
                if (currUploaded != null 
                        && (    maxUploadedJoin == null 
                        || currUploaded.doubleValue() > maxUploadedJoin.getDocumentVersionUploaded().doubleValue())) {
                    maxUploadedJoin = currJoin;
                }                    
            }            
        }
        return maxUploadedJoin;
    }
    
    // ========================== DocumentVersion ==============================
    
    public Set<Long> getVersionIdsByDocumentId(Long documentId) {
        Set<Long> versionIds = new HashSet<Long>();
        List<T> joins = this.documentIdToJoins.get(documentId);
        for (T join : joins) {
            Long documentVersionId = join.getDocumentVersionId();
            if (documentVersionId != null) {
                versionIds.add(documentVersionId);
            }
        }
        return versionIds;
    }
    
    public Set<Long> getAllDocumentVersionIds() {
        return this.documentVersionIdToJoins.keySet();
    } 
    
    public T getSomeJoinForDocumentVersionId(Long documentVersionId) {
        if (this.documentVersionIdToJoins.containsKey(documentVersionId)) {
            List<T> joins = this.documentVersionIdToJoins.get(documentVersionId);
            return joins.get(0);
        } else {
            return null;
        }
    }    
    
    // ======================= ReviewCycleCellResult ===========================
    
    public Set<Long> getCellResultIdsByDocumentVersionId(Long documentVersionId) {
        Set<Long> cellResultIds = new HashSet<Long>();
        List<T> joins = this.documentVersionIdToJoins.get(documentVersionId);
        for (T join : joins) {
            Long cellResultId = join.getReviewCycleCellResultId();
            if (cellResultId != null) {
                cellResultIds.add(cellResultId);
            }
        }
        return cellResultIds;
    }    
    
    public T getSomeJoinForCellResultId(Long cellResultId) {
        if (this.cellResultIdToJoins.containsKey(cellResultId)) {
            List<T> joins = this.cellResultIdToJoins.get(cellResultId);
            return joins.get(0);
        } else {
            return null;
        }
    }     
    
    // ======================= ReviewCycleNodeResult ===========================
    
    public Set<Long> getNodeResultIdsByCellResultId(Long cellResultId) {
        Set<Long> nodeResultIds = new HashSet<Long>();
        List<T> joins = this.cellResultIdToJoins.get(cellResultId);
        for (T join : joins) {
            Long nodeResultId = join.getReviewCycleNodeResultId();
            if (nodeResultId != null) {
                nodeResultIds.add(nodeResultId);
            }
        }
        return nodeResultIds;
    }     
    
    public T getSomeJoinForNodeResultId(Long nodeResultId) {
        if (this.nodeResultIdToJoins.containsKey(nodeResultId)) {
            List<T> joins = this.nodeResultIdToJoins.get(nodeResultId);
            return joins.get(0);
        } else {
            return null;
        }
    }     
    
    public List<T> getJoinsByNodeResultId(Long nodeResultId) {
        if (this.nodeResultIdToJoins.containsKey(nodeResultId)) {
            return this.nodeResultIdToJoins.get(nodeResultId);
        } else {
            return new ArrayList<T>();
        }
    }
    
    public T getNodeResultJoinByVersionNodeAndMaxArrivalDate(Long documentVersionId, Long nodeId) {
	    // Frühe Exits: ohne nodeId oder ohne Trefferliste kein Ergebnis
	    if (nodeId == null) return null;

	    List<T> joins = this.documentVersionIdToJoins.get(documentVersionId);
	    if (joins == null || joins.isEmpty()) return null;

	    T maxArrivalJoin = null;
	    Double maxArrival = null;

	    for (T join : joins) {
	        if (join == null) continue;

	        Long currNodeId = join.getReviewCycleNodeResultReviewCycleNodeId();
	        if (!Objects.equals(currNodeId, nodeId)) continue; // null-sicherer Vergleich

	        Double currArrival = join.getReviewCycleNodeResultArrivalDate();
	        // Nur gültige (nicht-null, nicht-NaN/∞) Ankunftsdaten berücksichtigen
	        if (currArrival == null || currArrival.isNaN() || currArrival.isInfinite()) continue;

	        if (maxArrivalJoin == null || maxArrival == null || Double.compare(currArrival, maxArrival) > 0) {
	            maxArrivalJoin = join;
	            maxArrival = currArrival;
	        }
	    }
	    return maxArrivalJoin;
    	}
    
    // ===================== ReviewCyclePositionResult =========================
    
    public T getPositionResultJoinByVersionWithoutCommentReferenceAndMaxResultDate(Long documentVersionId) {
        List<T> joins = this.documentVersionIdToJoins.get(documentVersionId);
        if (joins == null) {
            return null;
        }
        
        T maxResultJoin = null; 
        for (T join : joins) {
            Long commentReferenceId = join.getReviewCyclePositionResultCommentReferenceId();
            if (commentReferenceId == null) {
                Double currResultDate = join.getReviewCyclePositionResultResultDate();
                Double maxResultDate = maxResultJoin != null ? maxResultJoin.getReviewCyclePositionResultResultDate() : null;
                if (currResultDate != null
                        && (maxResultDate == null || currResultDate.doubleValue() > maxResultDate.doubleValue())) {
                    maxResultJoin = join;
                }
            }
        }
        return maxResultJoin;
    }
}
