/***********************************************************
* $Id$
*
* Copyright (C) 2017 ev-i Informationstechnologie Gmbh
*
**********************************************************/

define([ 
        "cdes/util/I18nHelper",
        "cdes/util/NameHelper",
    "cdes/widget/base/ListWidget",
    "clazzes/TinyLog",
    "clazzes/dateTime/DateHelper",
    "clazzes/form/FancyButton",
    "clazzes/topic",
    "clazzes/util/DOMHelper",
    "clazzes/util/ErrorHelper",
    "clazzes/widgets/layout/InfoDialog",
    "dojo/dom-class",
    "dojo/dom-construct",
    "dojo/dom-style",
    "dojo/on",
    "dojo/string",
    "dojo/_base/declare",
    "dojo/_base/lang",
    "dojo/i18n!/cdes/nls/cdes-web-i18n.js"
], function(
            I18nHelper,
            NameHelper,
    ListWidget,
    TinyLog,
    DateHelper,
    FancyButton,
    topic,
    DOMHelper,
    ErrorHelper,
    InfoDialog,
    domClass,
    domConstruct,
    domStyle,
    on,
    string,    
    declare,
    lang,
    i18n
) {

    var className = "at.cdes.review.ReviewProtocolInvalidationListWidget";

    var log = new TinyLog(className);

    var ReviewProtocolInvalidationListWidget = declare(className, ListWidget, {

        constructor : function(params) {
            lang.mixin(this, params);

            this.topDiv = this.constructTopDiv();

            this.allFieldsValid = true;
        },

        getWidgetId : function() {
            return "ReviewProtocolInvalidationListWidget";
        },

        getDataId : function() {
            return null;
        },

        getContainer : function() {
            return this.topDiv;
        },

        columnWidthKey : "reviewProtocolInvalidationList/columnWidths",

        getLocalStorageContextKeys : function() {
            var organisationPersonId = this.applicationContext.getPageContextOrganisationPersonId();
            var networkId = this.applicationContext.getPageContextNetworkId();
            return [organisationPersonId, networkId];
        },

        constructColumns : function() {
            var columns = [];

	    columns.push({
		     field : "_person",
		        id : "person",
                renderCell : lang.hitch(this, this.renderPerson),
		  sortable : true,
		     label : i18n.reviewProtocolInvalidationPersonCaption
	    });
	    columns.push({
		    field : "_resultDate",
		       id : "resultDate",
                formatter : lang.hitch(this, this.formatResultDate),
		 sortable : true,
		    label : i18n.reviewProtocolInvalidationResultDateCaption
	    });
	    columns.push({
		     field : "_comment",
		        id : "comment",
                renderCell : lang.hitch(this, this.renderComment),                
		  sortable : true,
		     label : i18n.reviewProtocolInvalidationCommentCaption
	    });
	    columns.push({
		     field : "_resultOptionName",
			id : "reviewCycleResultOptionName",
                renderCell : lang.hitch(this, this.renderResultOptionName),
		  sortable : true,
		    label : i18n.reviewProtocolInvalidationResultOptionCaption
	    });            

            return columns;            
        },

        getInvalidatorMainPersonString : function(invalidationResultJoin) {
            var personCommonName = NameHelper.getPersonCommonName(invalidationResultJoin, "positionMainPerson");
            var result = string.substitute(i18n.deputyPerson, {
                person : personCommonName
            });
            return result;            
        },

        getInvalidatorMainPersonWithOrganisationString : function(invalidationResultJoin) {
            var organisationName = invalidationResultJoin.positionMainOrganisationName;
            var personCommonName = NameHelper.getPersonCommonName(invalidationResultJoin, "positionMainPerson");
            var result = string.substitute(i18n.deputyPersonWithOrganisation, {
                person : personCommonName,
                organisation : organisationName                
            });
            return result;            
        },

        getInvalidatorPersonString : function(invalidationResultJoin) {
            var organisationName = invalidationResultJoin.positionOrganisationName;
            var personCommonName = NameHelper.getPersonCommonName(invalidationResultJoin, "positionPerson");
            var result = string.substitute(i18n.organisationCommonName, {
                organisationName : organisationName,
                personCommonName : personCommonName                
            });
            return result;
        },            

        renderPerson : function(invalidationResultJoin, data, cell) {
            var cellDiv = domConstruct.create("div", null, null);
            DOMHelper.createTextNode("div", this.getInvalidatorPersonString(invalidationResultJoin), cellDiv);
            if (!invalidationResultJoin.positionParticipantMainParticipantFlag || invalidationResultJoin.positionParticipantInactiveFlag) {
			    if (invalidationResultJoin.positionOrganisationId != invalidationResultJoin.positionMainOrganisationId) {
	                DOMHelper.createTextNode("div", this.getInvalidatorMainPersonWithOrganisationString(invalidationResultJoin), cellDiv, "reviewProtocolInvalidationSubstitute");
	            } else if (invalidationResultJoin.positionPersonId != invalidationResultJoin.positionMainPersonId) {
	                DOMHelper.createTextNode("div", this.getInvalidatorMainPersonString(invalidationResultJoin), cellDiv, "reviewProtocolInvalidationSubstitute");
	            }
            }
            return cellDiv;            
        },            

        formatResultDate : function(_resultDate, invalidationResultJoin) {
            var resultDate = invalidationResultJoin.reviewCyclePositionResultResultDate;
    	    if (resultDate != null) {
    		return DateHelper.formatUtcSecondsWithTimeZone(resultDate, this.applicationContext.getTimeZone(), i18n.dateWithSecondsPattern); 
    	    } else {
    		return "";
    	    }            
        },

        renderComment : function(invalidationResultJoin, data, cell) {
            var comment = invalidationResultJoin.reviewCyclePositionResultComment;
            var isAuthorized = invalidationResultJoin.reviewCyclePositionResultIsAuthorized;            
            
            var cellDiv = domConstruct.create("div", null, null);
            if (comment != null) {
                // Safe html thanks to server side
	        var node = domConstruct.create("div", { innerHTML : comment });
                domConstruct.place(node, cellDiv);                
//                DOMHelper.createTextNode("div", comment, cellDiv);
            }
            if (isAuthorized) {
                DOMHelper.createTextNode("div", i18n.reviewCyclePositionResultAuthorized, cellDiv, "reviewProtocolInvalidationAuthorized");
            }
            return cellDiv;            
        },            

        getResultOptionName : function(invalidationResultJoin) {
            var reviewCycleResultOptionId = invalidationResultJoin.reviewCyclePositionResultReviewCycleResultOptionId;
            if (this.reviewContext != null && reviewCycleResultOptionId != null) {
                var reviewCycleResultOption = this.reviewContext.getResultOption(reviewCycleResultOptionId);
                if (reviewCycleResultOption != null && reviewCycleResultOption.name != null) {
                    var locale = this.applicationContext.getPageContextPersonVariablesUserLocale();
                    var reviewCycleResultOptionName = I18nHelper.getCurrentLocaleString(reviewCycleResultOption.name, locale);
                    return reviewCycleResultOptionName;
                } else {
                    return "";
                }                    
            } else {
                return "";
            }                
        },

        renderResultOptionName : function(invalidationResultJoin, data, cell) {
            var cellDiv = domConstruct.create("div", null, null);
            var resultOptionName = this.getResultOptionName(invalidationResultJoin);
            DOMHelper.createTextNode("span", resultOptionName, cellDiv);

            /* Outcommented since deletion on server side not yet implemented in a reliable way.            
            if (this.reviewContext.hasGlobalAction("superAdminRight")) {
                var deleteInvalidationResultButton = this.constructDeletePositionResultButton(invalidationResultJoin);
                domConstruct.place(deleteInvalidationResultButton.domNode, cellDiv);
            }
            */            

            return cellDiv;            
        },            

        constructDeletePositionResultButton : function(positionResultJoin) {
	    var button = new FancyButton({
		    title : i18n.reviewProtocolDeletePositionResultButtonToolTip,
		iconClass : "fancyButtonIcon17x18 fancyButton17x18 deleteButton"
	    });

	    domClass.add(button.domNode, "listButton reviewProtocolInvalidationDeleteButton");

	    on(button, "click", lang.hitch(this, function() {
		var message = string.substitute(i18n.reviewProtocolDeletePositionResultDialogQuestion, {
                             resultDate : this.formatResultDate(null, positionResultJoin),
		    documentVersionName : positionResultJoin.documentVersionName
		});

		InfoDialog.showQuestion({
		      title : i18n.reviewProtocolAskDeletePositionResultDialogCaption,
		    message : message,
		    buttons : [
			{ type : InfoDialog.Button.YES, fct : lang.hitch(this, function() {
                            this.doDeletePositionResult(positionResultJoin);
			})},    			           
			{ type : InfoDialog.Button.NO }
		    ]
		}); 
	    }));

	    return button;
        },            

        doDeletePositionResult : function(positionResultJoin) {
	    var documentService = this.applicationContext.getService("documentService");
	    this.registerAsyncOperationStarted(ReviewProtocolInvalidationListWidget.AsyncOperation.DELETE);

	    var organisationPersonId = this.applicationContext.getPageContextOrganisationPersonId();
	    documentService.deletePositionResult(organisationPersonId, positionResultJoin.reviewCyclePositionResultId).then(
		lang.hitch(this, function(result) {
		    this.registerAsyncOperationFinished(ReviewProtocolInvalidationListWidget.AsyncOperation.DELETE);

                    if (result) {
                        var message = string.substitute(i18n.reviewProtocolDeletePositionResultSuccessful, {
                                     resultDate : this.formatResultDate(null, positionResultJoin),
		            documentVersionName : positionResultJoin.documentVersionName
                        });                        
                        
                        topic.publish("message/ok", message);
                        // TODO reload etc.
                    } else {
                        var errorMessage = string.substitute(i18n.reviewProtocolDeletePositionResultNotPossible, {
                                     resultDate : this.formatResultDate(null, positionResultJoin),
		            documentVersionName : positionResultJoin.documentVersionName
                        });
                        topic.publish("message/error", errorMessage, true);                        
                    }                        
		}),
		lang.hitch(this, function(err) {
		    ErrorHelper.processAsyncError({
				   err : err,
				widget : this,
			asyncOperation : ReviewProtocolInvalidationListWidget.AsyncOperation.DELETE,
				opName : "deletePositionResult",
			       message : i18n.reviewProtocolPositionResultDeleteFailed
		    });
		})).otherwise(
		    lang.hitch(this, function(err) {
			log.error("Error while calling function [deletePositionResult]", err);
		    }));
        },

        constructTopDiv : function() {
            var topDiv = domConstruct.create("div", null, null);


	    var grid = this.constructGrid({
		idProperty : "reviewCyclePositionResultId", 
		    gridId : "ReviewProtocolInvalidationGrid",
                 gridClass : "Grid"
	    });
            domConstruct.place(grid, topDiv);

            return topDiv;
        },

        setData : function(params) {
            this.invalidationResultJoins = params.invalidationResultJoins;
            this.reviewContext = params.reviewContext;

            this.updateAuxiliaryData();

            this.reload();
        },

        reload : function() {
            this.setStoreData(this.invalidationResultJoins);
            this.grid.resize();
        },

	updateAuxiliaryData : function() {
	    for (var n = 0; n < this.invalidationResultJoins.length; n++) {
			var invalidationResultJoin = this.invalidationResultJoins[n];
            invalidationResultJoin._person = this.getInvalidatorPersonString(invalidationResultJoin)
            if (!invalidationResultJoin.positionParticipantMainParticipantFlag || invalidationResultJoin.positionParticipantInactiveFlag) {
				if (invalidationResultJoin.positionOrganisationId != invalidationResultJoin.positionMainOrganisationId) {
                	invalidationResultJoin._reviewer += this.getInvalidatorMainPersonWithOrganisationString(invalidationResultJoin);
	            } else if (invalidationResultJoin.positionPersonId != invalidationResultJoin.positionMainPersonId) {
                	invalidationResultJoin._reviewer += this.getInvalidatorMainPersonString(invalidationResultJoin);
	            }
            }    
            
            invalidationResultJoin._resultDate = this.formatResultDate(null, invalidationResultJoin);

            var comment = invalidationResultJoin.reviewCyclePositionResultComment;
            var isAuthorized = invalidationResultJoin.reviewCyclePositionResultIsAuthorized;            
            invalidationResultJoin._comment = (comment != null ? comment : "") + (isAuthorized ? i18n.reviewCyclePositionResultAuthorized : "");
            invalidationResultJoin._resultOptionName = this.getResultOptionName(invalidationResultJoin);
	    }
	},

        updateWidgetState : function() {

        },

        resize : function(newSize) {
            // We need to perform the resize in two steps.
            // First set the desired width, and some arbitrary height.
            var width = newSize.w + 0.01;
            var height = this.getActualHeight();
            this.grid.resize({ w : width, h : height });
        },

        destroy : function() {
            this.inherited(arguments);
            this.grid.destroy();
        }
    });

    ReviewProtocolInvalidationListWidget.AsyncOperation = {
        DELETE : "Delete"
    };        

    return ReviewProtocolInvalidationListWidget;
});
