/***********************************************************
* $Id$
*
* Copyright (C) 2017 ev-i Informationstechnologie Gmbh
*
**********************************************************/

define([ 
    "clazzes/TinyLog",
    "clazzes/canvas/CanvasHelper",
    "clazzes/form/FancyButton",
    "clazzes/mouse/LocalDragOperation",
    "clazzes/mouse/MouseHelper",
    "clazzes/widgets/layout/ContentWidget",
    "dojo/_base/declare",
    "dojo/_base/lang", 
    "dojo/dom-class",
    "dojo/dom-construct",
    "dojo/dom-prop",
    "dojo/dom-style",
    "dojo/io-query",
    "dojo/on",
    "dojo/i18n!/cdes/nls/cdes-web-i18n.js"    
], function(
    TinyLog,
    CanvasHelper,
    FancyButton,
    LocalDragOperation,
    MouseHelper,
    ContentWidget,
    declare,
    lang,
    domClass,
    domConstruct,
    domProp,
    domStyle,
    ioQuery,
    on,    
    i18n    
) {

    var className = "at.cdes.web.document.DocumentCompareOverviewWidget";

    var log = new TinyLog(className);

    var DocumentCompareOverviewWidget = declare(className, ContentWidget, {

        constructor : function(params) {
            lang.mixin(this, params);

            this.contentDiv = domConstruct.create("div", null, null);
            domClass.add(this.contentDiv, "documentCompareOverviewContentDiv");            

            this.constructCanvas();
            this.constructShowHideIcon();
            this.constructRotateRightIcon();
            this.constructRotateLeftIcon();            

            this.updateWidgetState();            

	    this.mouseDownHandler = {
		handleEvent : lang.hitch(this, this.mouseDownHandlerFunction)
	    };
	    this.contentDiv.addEventListener("mousedown", this.mouseDownHandler, false);

	    this.mouseMoveHandler = {
		handleEvent : lang.hitch(this, this.mouseMoveUpHandlerFunction)
	    };
	    this.contentDiv.addEventListener("mousemove", this.mouseMoveHandler, false);
	    this.mouseUpHandler = {
		handleEvent : lang.hitch(this, this.mouseMoveUpHandlerFunction)
	    };
	    this.contentDiv.addEventListener("mouseup", this.mouseUpHandler, false);
        },

        constructCanvas : function() {
            this.canvas = CanvasHelper.getMostNativeCanvasInstance();
            domClass.add(this.canvas.domNode, "fixedDialogWidget documentCompareOverviewCanvas");
            domConstruct.place(this.canvas.domNode, this.contentDiv);                        
        },

        constructShowHideIcon : function() {
            this.showOverview = true;
            this.showHideIcon = domConstruct.create("div", null, null);
            domClass.add(this.showHideIcon, "fixedDialogWidget fancyButtonIcon17x18 hideOverview documentCompareOverviewShowHideButton");
            domProp.set(this.showHideIcon, "title", i18n.documentCompareOverviewHideToolTip);
            domConstruct.place(this.showHideIcon, this.contentDiv);
            
            on(this.showHideIcon, "click", lang.hitch(this, function() {
                this.showOverview = !this.showOverview;
                this.updateWidgetState();                
//                on.emit(this, "needResize");
            }));
        },

        constructRotateRightIcon : function() {
            this.rotateRightIcon = domConstruct.create("div", null, null);
            domClass.add(this.rotateRightIcon, "fixedDialogWidget documentCompareOverviewRotateRightButton");
            domProp.set(this.rotateRightIcon, "title", i18n.documentCompareOverviewRotateRightToolTip);
            domConstruct.place(this.rotateRightIcon, this.contentDiv);
            
            on(this.rotateRightIcon, "click", lang.hitch(this, function() {
                this.tiledImage.rotate(270);
                on.emit(this, "needResize");
            }));            
        },

        constructRotateLeftIcon : function() {
            this.rotateLeftIcon = domConstruct.create("div", null, null);
            domClass.add(this.rotateLeftIcon, "fixedDialogWidget documentCompareOverviewRotateLeftButton");
            domProp.set(this.rotateLeftIcon, "title", i18n.documentCompareOverviewRotateLeftToolTip);
            domConstruct.place(this.rotateLeftIcon, this.contentDiv);
           
            on(this.rotateLeftIcon, "click", lang.hitch(this, function() {
                this.tiledImage.rotate(90);
                on.emit(this, "needResize");
            }));            
        },

        getWidgetId : function() {
            return "DocumentCompareOverviewWidget";
        },

        getDataId : function() {
            return null;
        },

        getContainer : function() {
            return this.contentDiv;
        },

        resize : function(newSize) {
            this.width = newSize.w;
            this.height = newSize.h;            
            domStyle.set(this.contentDiv, "width", (this.width + 10) + "px");
            domStyle.set(this.contentDiv, "height", (this.height + 10) + "px");

            this.canvas.resize(newSize);
        },

        setPosition : function(relativeX, relativeY, relativeWidth, relativeHeight) {
            this.position = {
                     x : relativeX * this.width,
                     y : relativeY * this.height,
                 width : relativeWidth * this.width,
                height : relativeHeight * this.height
            };
        },

        getWidth : function() {
            return this.width;
        },

        getHeight : function() {
            return this.height;
        },        


        getDesiredWidth : function() {
            if (this.tiledImage == null) {
                return 320;
            }                
            
            var rectangularAngle = this.tiledImage != null ? this.tiledImage.getRectangularAngle() : 0;
            var width = this.tiledImage.getWidth();
            var height = this.tiledImage.getHeight();            
//            var width = rectangularAngle == 1 || rectangularAngle == 3 ? this.tiledImage.getWidth() : this.tiledImage.getHeight();
//            var height = rectangularAngle == 1 || rectangularAngle == 3 ? this.tiledImage.getHeight() : this.tiledImage.getWidth();
            var dimensionFraction = width / height;
            log.info("angle = [" + rectangularAngle + "], desiredWidth: width = [" + width + "], height = [" + height + "], dimensionFraction = [" + dimensionFraction + "]");
            var retValue = null;
            if (dimensionFraction > 1) {
                // Width is greater than height, thus set width to 320, the height will get the corresponding smaller value
                retValue = 320;
            } else {
                retValue = dimensionFraction * 320;
            }            

            log.info("==> Setting desiredWidth = [" + retValue + "]");            
            return retValue;            
//            var rectangularAngle = this.tiledImage != null ? this.tiledImage.getRectangularAngle() : 0;
//            return rectangularAngle == 1 || rectangularAngle == 3 ? 80 : 320;
        },        

        getDesiredHeight : function() {
            if (this.tiledImage == null) {
                return 80;
            }                
            
            var rectangularAngle = this.tiledImage != null ? this.tiledImage.getRectangularAngle() : 0;
            var width = this.tiledImage.getWidth();
            var height = this.tiledImage.getHeight();            
//            var width = rectangularAngle == 1 || rectangularAngle == 3 ? this.tiledImage.getWidth() : this.tiledImage.getHeight();
//            var height = rectangularAngle == 1 || rectangularAngle == 3 ? this.tiledImage.getHeight() : this.tiledImage.getWidth();
            var dimensionFraction = width / height;
            log.info("angle = [" + rectangularAngle + "], desiredHeight: width = ["
                + width + "], height = [" + height + "], dimensionFraction = [" + dimensionFraction + "]");
            var retValue = null;
            if (dimensionFraction < 1) {
                // Height is greater than width, thus set height to 320, the width will get the corresponding smaller value
                retValue = 320;
            } else {
                retValue = 320 / dimensionFraction;
            }            

            log.info("==> Setting desiredHeight = [" + retValue + "]");            
            return retValue;            
//            var rectangularAngle = this.tiledImage != null ? this.tiledImage.getRectangularAngle() : 0;
//            return rectangularAngle == 1 || rectangularAngle == 3 ? 320 : 80;
        },

        setData : function(params) {
            lang.mixin(this, params);            
            this.reload();            
        },

        reload : function() {
            // TODO Case ownDocumentVersion / chosenDocumentVersion is null

            if (this.imageObject == null) {
                // create new thumbnail image
                this.imageObject = new Image();

                this.imageObject.onload = lang.hitch(this, function() {
                    this.imageLoadFailed = false;
                    this.imageLoaded = true;
                    this.repaint();
                });
                this.imageObject.onabort = lang.hitch(this, function() {
                    this.imageLoadFailed = true;
                    this.repaint();
                });
                this.imageObject.onerror = lang.hitch(this, function() {
                    this.imageLoadFailed = true;
                    this.repaint();
                });

                var ownDocumentVersionId = this.ownDocumentVersion.id;
                var chosenDocumentVersionId = this.chosenDocumentVersion.id;                
                var thumbSize = this.tiledImage.getThumbSize();
                var angle = this.tiledImage.getAngle();

                var paramString = ioQuery.objectToQuery({
                    service : "CDESMimeService/4/compareDocumentVersionPngMimeSource",
                         sp : [ ownDocumentVersionId, chosenDocumentVersionId, thumbSize, "T", angle ],
                         ts : dojoConfig.tabSessionId                    
                });                    

                this.imageObject.src = "/cdes/app?" + paramString;

                // TODO CWH.inhibitSelection(img);

                /* TODO: 

                loadThumb(img,w,h,angle);

                implemented as: 

                var thumb = img.parentNode;
                var overview = thumb.parentNode;

                thumb.style.width = w + 'px';
                thumb.style.height = h + 'px';

                var viewport=document.getElementById('viewport');

                var x = parseInt(viewport.style.left);
                var y = parseInt(viewport.style.top);
                var vw = parseInt(viewport.style.width);
                var vh = parseInt(viewport.style.height);
                
                var ow = w+2;
                var oh = h+2;

                x += vw - ow - 8;
                y += vh - oh - 8;

                overview.style.left = x + 'px';
                overview.style.top = y + 'px';
                overview.style.width = ow + 'px';
                overview.style.height = oh + 'px';
                overview.style.zIndex = 97;
                */
            }            
        },

        dropImage : function() {
            delete this.imageObject;
        },

        repaint : function() {
            var context = this.canvas.getContext2D();
            context.clear();
            
            if (this.imageObject && this.imageLoaded) {
                var thumbWidth = this.tiledImage.getThumbWidth();
                var thumbHeight = this.tiledImage.getThumbHeight();

                context.drawImage(this.imageObject, 0, 0, thumbWidth, thumbHeight);

                try {
                    context.save();
                    context.setFillStyle("rgba(255,0,255,0.4)");
                    context.rect(this.position.x, this.position.y, this.position.width, this.position.height);
                    context.fill();
                } finally {
                    context.restore();
                }                    
            } else {
                var width = this.canvas.get("width");
                var height = this.canvas.get("height");
                context.setFont("normal 12px sans-serif");
                context.setTextAlign("center");
                
                var message = "";
                if (this.imageObject) {
                    if (this.imageLoadFailed) {
                        message = i18n.documentCompareThumbImageDisplayFailed;
                    } else {
                        message = i18n.documentCompareThumbImageEditImageLoading;
                    }                                   
                } else {
                    message = i18n.documentCompareThumbImageEditNoImageLoaded;
                }
                if (this.imageLoadFailed)
					window.alert(message);
                else
                	context.fillText(message, width / 2, height / 2, width);                                        
            }            
        },

        updateWidgetState : function() {
            if (this.showOverview) {
                domProp.set(this.showHideIcon, "title", i18n.documentCompareOverviewHideToolTip);
                domClass.replace(this.showHideIcon, "hideOverview", "showOverview");
                domClass.remove(this.canvas.domNode, "hidden");
                domClass.remove(this.rotateRightIcon, "hidden");
                domClass.remove(this.rotateLeftIcon, "hidden");
                domClass.add(this.contentDiv, "documentCompareOverviewContentDiv");
            } else {
                domProp.set(this.showHideIcon, "title", i18n.documentCompareOverviewShowToolTip);
                domClass.replace(this.showHideIcon, "showOverview", "hideOverview");                
                domClass.add(this.canvas.domNode, "hidden");
                domClass.add(this.rotateRightIcon, "hidden");
                domClass.add(this.rotateLeftIcon, "hidden");
                domClass.remove(this.contentDiv, "documentCompareOverviewContentDiv");                
            }
        },

    	mouseDownHandlerFunction : function(mouseEvent) {
    	    if (MouseHelper.isLeftButtonPressed(mouseEvent) || MouseHelper.isMiddleButtonPressed(mouseEvent)) {
        	mouseEvent.preventDefault();
        	mouseEvent.stopPropagation();
        	
                // Subtract empty padding space at the upper and left border.                
                var mouseX = mouseEvent.offsetX;// - this.contentDiv.offsetLeft;
                var mouseY = mouseEvent.offsetY;// - this.contentDiv.offsetTop;

                // Only accept mouse clicks on the highlighted area.
                if (this.isOverHighlightedArea(mouseX, mouseY)) {
        	    this.dragOperation = new LocalDragOperation(mouseEvent, {
    		           domNode : this.contentDiv,
    	                dragFilter : function() { return MouseHelper.isLeftButtonPressed(mouseEvent) || MouseHelper.isMiddleButtonPressed(mouseEvent); },
    	               performDrag : lang.hitch(this, this.performDrag),
    	               incremental : true,
                callPreventDefault : false                        
        	    });	
                }                    
            }
        },

        isOverHighlightedArea : function(mouseX, mouseY) {
            return !(this.position == null
                || mouseX < this.position.x || mouseX > this.position.x + this.position.width
                || mouseY < this.position.y || mouseY > this.position.y + this.position.height);
        },            

    	performDrag : function(mouseDx, mouseDy, mousePosition, event) {
            var widthFactor = this.width / this.tiledImage.getWidth();
            var heightFactor = this.height / this.tiledImage.getHeight();
            var dx = mouseDx / widthFactor;
            var dy = mouseDy / heightFactor;
            this.tiledImage.move(-dx, -dy);
        },

        mouseMoveUpHandlerFunction : function(mouseEvent) {
            var mouseX = mouseEvent.offsetX;
            var mouseY = mouseEvent.offsetY;

            var cursor = null;
            if (this.isOverHighlightedArea(mouseX, mouseY)) {
                if (this.dragOperation != null && this.dragOperation.isActive()) {
                    cursor = "all-scroll";
                } else {
                    cursor = "pointer";
                }     
            } else {
                cursor = "default";
            }                    

            if (log.isDebugEnabled()) {
                log.debug("Setting mouse cursor to [" + cursor + "]");
            }                

            domStyle.set(this.contentDiv, "cursor", cursor);
        }        
    });

    return DocumentCompareOverviewWidget;
});
