/***********************************************************
* $Id$
*
* Copyright (C) 2017 ev-i Informationstechnologie Gmbh
*
**********************************************************/


define([
    "cdes/util/ColumnHelper",
    "cdes/util/FancyColumnResizer",
    "cdes/util/FancyMemory",
    "clazzes/TinyLog",
    "clazzes/dateTime/DateHelper",
    "clazzes/util/StringHelper",
    "clazzes/widgets/layout/ContentWidget",
    "dgrid/Grid",
    "dgrid/OnDemandGrid",
    "dgrid/Selection",
    "dgrid/extensions/DijitRegistry",
    "dgrid/util/misc",
    "dojo/_base/declare",
    "dojo/_base/lang",
    "dojo/dom-class",
    "dojo/has",
    "dojo/on",
    "dstore/Memory",
    "dstore/Trackable",
    "dojo/i18n!/cdes/nls/cdes-web-i18n.js"],
function(
    ColumnHelper,
    FancyColumnResizer,
    FancyMemory,
    TinyLog,
    DateHelper,
    StringHelper,
    ContentWidget,
    Grid,
    OnDemandGrid,
    Selection,
    DijitRegistry,
    dgridMiscUtil,          
    declare,
    lang,
    domClass,
    has,
    on,
    Memory,
    Trackable,
    i18n) {
    
    var className = "at.cdes.web.widget.base.ListWidget";

    var log = new TinyLog(className);

    var ListWidget = declare(className, ContentWidget, {

        constructor : function() {
            this.columnSettings = new Object();

            /* Dynamic column hiding based on the ColumnHider plugin of dgrid, but with own CheckBoxes outside the table */ 
            this.columnVisibility = new Object();
            this.columnHiderRules = new Object();           

            // On the very first reload, column widths are restored from local storage (if columnWidthKey is set).
            this.columnWidthsSet = false;
        },

        dataAttributeName : null,
        // summary:
        //     The name of the attribute (of the this pointer), where the list of objects injected into the store / the grid is stored. 

        constructColumns : function() {
            // summary:
            //     Returns the columns of the grid.  Needs to be implemented in subclass.           
        },          

        updateIndividualColumns : function() {
            // summary:
            //     Contains the updateColumn statements, like
            //     this.updateColumn("personId", this.columnSettings.showName);     
            //     Should be implemented in subclass, if one wants columns being configurable using CheckBoxes.     
        },          

        constructGrid : function(params) {
            // summary:
            //     Constructs the grid instance for this ListWidget.
            // description:
            //     gridClass specifies, which of the dgrid classes is to be used.
            //     'OnDemandGrid' creates an OnDemandGrid, and is quite well tested, i.e. should usually be used.
            //     'Grid' creates a Grid.  The purpose of offering this possibility is, that a Grid renders all
            //        of its rows at once.  Usually, when dealing with grids, this is disliked for reasons of
            //        efficiency, but if you want to base layout on the actual height of a non-scrollbar-grid,
            //        this actually is desired.  The Selection code wasn't yet tested for the Grid case, and
            //        quite possibly doesn't work.            

            this.idProperty = params.idProperty;
            var gridId = params.gridId;
            var idSuffix = params.idSuffix;
            this.selectionMode = params.selectionMode;
            this.gridClass = params.gridClass;
            var additionalModules = params.modules;            
            if (this.selectionMode == null) {
                this.selectionMode = "none";
            }
            if (this.gridClass == null) {
                this.gridClass = "OnDemandGrid";
            }                

            this.constructDataBackend();

            // Raised farOffRemoval to 20000, compare https://github.com/SitePen/dgrid/issues/1281
            // (the problem seems to be, that with the default farOffRemoval, and relatively high columns,
            // a near-to-endless recursion of "request additional data" and "delete it again" occurs)       

            var modules = [this.getGridClass(), FancyColumnResizer, DijitRegistry];
            if (this.selectionMode != "none") {
                modules.push(Selection);
            }
            if (additionalModules != null) {
                for (var n = 0; n < additionalModules.length; n++) {
                    modules.push(additionalModules[n]);
                }
            }                

            var gridParams = this.getGridParams();
            var MyGrid = declare(modules);
            this.grid = new MyGrid(gridParams);

            this.grid.afterResizeMouseUp = lang.hitch(this, this.handleColumnResize);

            var className = gridId.substring(0, 1).toLowerCase() + gridId.substring(1);
            domClass.add(this.grid.domNode, className);     

            this.grid.startup();
            this.gridRuleSelectorPrefix = "#" + dgridMiscUtil.escapeCssIdentifier(this.grid.domNode.id) + " .dgrid-column-";

            on(this.grid, "columnResizeMouseUp", lang.hitch(this, this.handleColumnResize));

            return this.grid.domNode;
        },

        getGridClass : function() {
            if (this.gridClass == "Grid") {
                return Grid;
            } else if (this.gridClass == "OnDemandGrid") {
                return OnDemandGrid;
            } else {
                throw new Error("Unsupported gridClass: [" + this.gridClass + "]");
            }
        },

        constructDataBackend : function() {
            if (this.gridClass == "Grid") {
                // Do nothing, data is passed to the Grid directly
            } else if (this.gridClass == "OnDemandGrid") {
                var TrackableMemory = declare([FancyMemory, Trackable]);
                this.store = new TrackableMemory({data : [], idProperty : this.idProperty});
            } else {
                throw new Error("Unsupported gridClass: [" + this.gridClass + "]");
            }
        },         

        getGridParams : function() {
            var gridParams = [];
            if (this.gridClass == "Grid") {
                gridParams = {
                               columns : this.constructColumns(),
                    keepScrollPosition : true // Not sure wether this one is supported by Grid - by OnDemandGrid, it is supported         
                };                    
            } else if (this.gridClass == "OnDemandGrid") {
                gridParams = {
                            collection : this.store,
                               columns : this.constructColumns(),
                    keepScrollPosition : true,
                         farOffRemoval : 20000          
                };
            } else {
                throw new Error("Unsupported gridClass: [" + this.gridClass + "]");
            }                

            if (this.selectionMode != "none") {
                gridParams.selectionMode = this.selectionMode;          
            }
            return gridParams;            
        },

        setStoreData : function(data) {
            if (this.gridClass == "Grid") {
                this.effectiveData = [];
                if (this.maxGridItems != null) {
                    for (var n = 0; n < data.length && n < this.maxGridItems; n++) {
                        this.effectiveData.push(data[n]);
                    }
                } else {
                    this.effectiveData = data;
                }                    

                this.grid.refresh();
                this.grid.renderArray(this.effectiveData);
            } else if (this.gridClass == "OnDemandGrid") {
                this.store.setData(data);
            } else {
                throw new Error("Unsupported gridClass: [" + this.gridClass + "]");
            }
        },            

        setColumns : function(columnSettings, touchedColumnId) {
            // summary:
            //     This is the PUBLIC function for changing the column settings from outside.

            this.updateColumns(columnSettings, touchedColumnId);
        },      

        setColumnSettings : function(columnSettings) {
            // summary:
            //     This is the PRIVATE function for actually setting the columnSettings map, by making a copy of it
            //     (updateColumns rely on the fact that this.columnSettings and columnSettings are not the same object)

            this.columnSettings = new Object();
            for (var column in columnSettings) {
                this.columnSettings[column] = columnSettings[column];
            }
        },

        updateColumns : function(columnSettings, touchedColumnId) {
            // summary:
            //     Updates the visible columns of the grid, while causing as little work for the browser as possible.
            // columnSettings:
            //     If null, this function assumes that the grid has been resetted with new data just now anyway,
            //     if not null, this object contains the new columnSettings that are set into this.columnSettings and
            //     processed by this function.

            // Record which of the columns corresponding to multiple CheckBoxes has changed data; the request to invalidate that
            // data has to be issued separately in case that the visibility of the column doesn´t change (e.g. switch from person + organisation to person)

            if (columnSettings) {
                // The case where this function is responsible for processing the new settings
                this.setColumnSettings(columnSettings);
            }               

            this.updateIndividualColumns();

            // If column width key is set, this ListWidget is configured to save its columnWidths.
            // At the very first time this function is called, we restore column widths from
            // local storage as far as possible.            
            if (this.columnWidthKey != null && !this.columnWidthsSet) {
                ColumnHelper.restoreColumnWidthsFromLocalStorage({
                    applicationContext : this.applicationContext,
                                  grid : this.grid,
                           contextKeys : this.getLocalStorageContextKeys(),
                              widthKey : this.columnWidthKey,
                              minWidth : this.minWidth != null ? this.minWidth : null
                });                

                this.columnWidthsSet = true;
            } else {
                if (touchedColumnId != null) {
                    var defaultWidth = this.getDefaultColumnWidth(touchedColumnId);
                    this.grid.resizeColumnWidth(touchedColumnId, defaultWidth);
                }                    
                
                ColumnHelper.cutColumnsToAvailableWidth(this.grid, touchedColumnId);
            }                
        },

        updateColumn : function(id, show, triggerStateChangeAlways) {
            show = !!show;

            var columnVisibility = (id in this.columnVisibility ? !!this.columnVisibility[id] : false);

            if (log.isDebugEnabled) {
                log.debug("Called updateColumn for id = [" + id + "], show = [" + show + "] and columnVisibility = [" + columnVisibility + "]");
            }           

            // Ignore visibility specifications for unknown columns.
            // Can e.g. happen, if visibility specifications are stored to local storage in browser,
            // and then a column is removed.        
            if (!(id in this.grid.columns)) {
                return;
            }           

            if (this.columnVisibility[id] == null || this.columnVisibility[id] != show) {
                // (Start) Code based on ColumnHider plugin for dgrid, dgrid/extensions/ColumnHider.js, _hideColumn, _showColumn, around line 300 (Start)

                if (show) {
                    if (log.isDebugEnabled()) {
                        log.debug("columnHiderRules[id] = " + (id in this.columnHiderRules ? this.columnHiderRules[id] : "---"));
                    }                   

                    if(id in this.columnHiderRules && this.columnHiderRules[id] != null) {
                        this.columnHiderRules[id].remove();

                        if (log.isDebugEnabled()) {
                            log.debug("Removed columnHiderRule [" + id + "]");
                        }                           

                        delete this.columnHiderRules[id];
                    }                                   

                    //this.grid.resizeColumnWidth(id, this.getDefaultColumnWidth(id));
                } else {
                    this.columnHiderRules[id] = dgridMiscUtil.addCssRule(this.gridRuleSelectorPrefix + dgridMiscUtil.escapeCssIdentifier(id, "-"), "display: none;");

                    if (log.isDebugEnabled())
                    	log.info("Added columnHiderRule [" + id + "]");

                    if((has("ie") === 8 || has("ie") === 10) && !has("quirks")){
                        var tableRule = dgridMiscUtil.addCssRule(".dgrid-row-table", "display: inline-table;");

                        window.setTimeout(lang.hitch(this, function(){
                            tableRule.remove();
                            this.grid.resize();
                        }), 0);
                    }                                   
                }

                this.columnVisibility[id] = show;

                // Update hidden state in actual column definition,
                // in case columns are re-rendered.                     
                this.grid.columns[id].hidden = !show;

                // Emit event to notify of column state change.
                on.emit(this.grid.domNode, "dgrid-columnstatechange", {
                       grid : this.grid,
                     column : this.grid.columns[id],
                     hidden : !show,
                    bubbles : true
                });

                // Adjust the size of the header.
                this.grid.resize();     

                // ((End) Code based on ColumnHider plugin (End))
            } else if (show && triggerStateChangeAlways) {
                on.emit(this.grid.domNode, "dgrid-columnstatechange", {
                       grid : this.grid,
                     column : this.grid.columns[id],
                     hidden : !show,
                    bubbles : true
                });                     
            }
        },

        handleColumnResize : function() {
            if (this.columnWidthKey != null) {
                var contextKeys = this.getLocalStorageContextKeys();
                ColumnHelper.saveColumnWidthsInLocalStorage({
                    applicationContext : this.applicationContext,
                                  grid : this.grid,
                           contextKeys : contextKeys,
                              widthKey : this.columnWidthKey                 
                });                 
            }             
        },

        restoreColumnWidths : function() {
            // Ensure that column widths stay constant
            if (this.columnWidthKey != null) {
                ColumnHelper.restoreColumnWidthsFromLocalStorage({
                    applicationContext : this.applicationContext,
                                  grid : this.grid,
                           contextKeys : this.getLocalStorageContextKeys(),
                              widthKey : this.columnWidthKey                    
                });                            
            }
        },

        getDefaultColumnWidth : function(columnId) {
            // summary:
            //     Returns the default width a column should have, if there is no width stored in local storage.            
            // description:
            //     Three cases:
            //        1. If the subclass defines a columnIdToDefaultWidth map, and the desired columnId has            
            //           an entry in that map, take that entry.  If a corresponding CSS class exists, the 
            //           entry should be in sync with it.  This case allows subclasses to specify the default 
            //           width of a newly shown column in detail.
            //        2. Otherwise, if the subclass defines a defaultColumnWidth, take it.              
            //        3. Otherwise, use some constant value as default column width.

            if (this.columnIdToDefaultWidth == null) {
                if (this.defaultColumnWidth == null) {
                    return 200;
                } else {
                    return this.defaultColumnWidth;
                }                    
            } else if (columnId in this.columnIdToDefaultWidth) {
                return this.columnIdToDefaultWidth[columnId];
            } else {
                if (this.defaultColumnWidth == null) {
                    return 200;
                } else {
                    return this.defaultColumnWidth;
                }
            }            
        },

        getTotalWidth : function() {
            return this.grid.headerNode.clientWidth;
        },

        getLocalStorageContextKeys : function() {
            throw new Error("Please implement getLocalStorageContextKeys; an implementation for this function is needed once this.columnWidthKey is not null.");
        },

        defaultBooleanFormatter : function(b, instance) {
            return b ? i18n.yes : i18n.no;
        },          

        dateFormatter : function(value, planningNotificationInfo) {
            if (value != null) {
                return DateHelper.formatUtcSecondsWithTimeZone(value, this.applicationContext.getTimeZone(), i18n.datePattern); 
            } else {
                return "";
            }
        },

        dateTimeFormatter : function(value, planningNotificationInfo) {
            if (value != null) {
                return DateHelper.formatUtcSecondsWithTimeZone(value, this.applicationContext.getTimeZone(), i18n.dateWithSecondsPattern); 
            } else {
                return "";
            }
        },

        dateFormatterMillis : function(value, planningNotificationInfo) {
            if (value != null) {
                return DateHelper.formatUtcMillisWithTimeZone(value, this.applicationContext.getTimeZone(), i18n.datePattern); 
            } else {
                return "";
            }
        },
        
        dateTimeFormatterMillis : function(value, planningNotificationInfo) {
            if (value != null) {
                return DateHelper.formatUtcMillisWithTimeZone(value, this.applicationContext.getTimeZone(), i18n.dateWithSecondsPattern); 
            } else {
                return "";
            }
        },

        filter : function(searchSpecs) {
            // summary:
            //    Contains the general code of a quick search (filter) on the data of this
            //    ListWidget.
            // searchSpecs:
            //    Either (in the simple case) a string, or an array of objects (searchString, tokenGetter, equalMatchRequired).
            //    A string s is translated into the object (s, this.getQuickSearchTokensForRow, false).
            //    For each object, a test wether any of the row tokens returned by the tokenGetter matches
            //    the given string is performed.  If equalMatchRequired is true, an exact match (after applying trim()) is required,
            //    otherwise it is sufficient if the given string is a substring of the token at hand.
            //    A data instance is filtered out if any of the checks fails.

            // Don´t filter too often.  If filtering starts while one is typing, and the list is big,
            // then the textbox freezes while one types until filtering has finished.
            if (this.filterHandle) {
                window.clearInterval(this.filterHandle);
                delete this.filterHandle;
            }

            if (typeof searchSpecs == "string") {
                searchSpecs = [{
                          searchString : searchSpecs,
                           tokenGetter : lang.hitch(this, this.getQuickSearchTokensForRow),
                    equalMatchRequired : false      
                }];                 
            }           

            this.filterHandle = window.setInterval(lang.hitch(this, function() {
                if (this[this.dataAttributeName] != null) {
                    this.startFilterUtcMillis = (new Date()).getTime();

                    // By default, take everything
                    var rowIdToFound = new Object();
                    for (var n = 0; n < this[this.dataAttributeName].length; n++) {
                        var rowId = this.getRowId(this[this.dataAttributeName][n]);
                        rowIdToFound[rowId] = true;             
                    }                   

                    for (var z = 0; z < searchSpecs.length; z++) {
                        var searchString = searchSpecs[z].searchString;
                        var tokenGetter = searchSpecs[z].tokenGetter;
                        var equalMatchRequired = searchSpecs[z].equalMatchRequired;             

                        var searchStrings = StringHelper.isEmpty(searchString) ? [] : searchString.split(",");
                        var trimmedSearchStrings = [];
                        for (var n = 0; n < searchStrings.length; n++) {
                            if (searchStrings[n] && searchStrings[n].trim().length > 0) {
                                trimmedSearchStrings.push(searchStrings[n].trim().toLowerCase());
                            }
                        }

                        for (var n = 0; n < this[this.dataAttributeName].length; n++) {
                            var rowId = this.getRowId(this[this.dataAttributeName][n]);
                            var rowTokens = tokenGetter(this[this.dataAttributeName][n]);
                            rowIdToFound[rowId] &= this.doRowTokensMatch(trimmedSearchStrings, rowTokens, equalMatchRequired);
                        }
                    }                   

                    var middleMillis = (new Date()).getTime();

                    var filteredDtos = [];
                    var numberOfNewFound = 0;
                    for (var n = 0; n < this[this.dataAttributeName].length; n++) {
                        var rowId = this.getRowId(this[this.dataAttributeName][n]);
                        if (rowIdToFound[rowId]) {
                            filteredDtos.push(this[this.dataAttributeName][n]);
                            numberOfNewFound++;
                        }
                    }

                    var difference = false;
                    var numberOfOldFound = 0;
                    for (var rowId in this.oldRowIdToFound) {
                        if (this.oldRowIdToFound[rowId]) {
                            numberOfOldFound++;
                            if (!(rowId in rowIdToFound) || !rowIdToFound[rowId]) {
                                difference = true;
                            }
                        }
                    }

                    difference |= (numberOfNewFound != numberOfOldFound);

                    this.oldRowIdToFound = new Object();
                    for (var rowId in rowIdToFound) {
                        this.oldRowIdToFound[rowId] = rowIdToFound[rowId];
                    }

                    if (difference || numberOfNewFound == 0) {
                        this.setStoreData(filteredDtos);
                        if (this.gridClass == "OnDemandGrid") {
                            this.grid.refresh();
                        }                        
                    }

                    var endMillis = (new Date()).getTime();
                }

                window.clearInterval(this.filterHandle);
                delete this.filterHandle;
            }), 0);
        },

        resetQuickSearch : function() {
            this.oldRowIdToFound = new Object();
        },

        doRowTokensMatch : function(searchStrings, tokens, equalMatchRequired) {
            var allFound = true;

            for (var z = 0; z < searchStrings.length; z++) {

                var found = false;
                for (var n = 0; n < tokens.length; n++) {
                    if (tokens[n] != null) {
                        if (!(typeof tokens[n] == "string")) {
                            tokens[n] = tokens[n].toString();
                        }       

                        tokens[n] = tokens[n].toLowerCase();

                        // TODO: Escape such that e.g. ":" can be searched properly (currently, searching for 17:04:26 gives no result even if
                        // there is such a timestamp
                        if (equalMatchRequired) {
                            if (tokens[n] == searchStrings[z]) {
                                found = true;
                                break;                          
                            }                           
                        } else {
                            if (tokens[n].indexOf(searchStrings[z]) != -1) {
                                found = true;
                                break;
                            }
                        }                               
                    }                                           
                }

                if (!found) {
                    allFound = false;
                    break;
                }
            }

            return allFound;
        },

        getQuickSearchTokensForRow : function(joinDto) {
            // summary:
            //     Returns all tokens quick search should use for the test, wether the given joinDto
            //     matches the current quick search string.         
        },          

        getSelectedItems : function() {
            var selectedItems = [];
            var selection = this.grid.selection;
            for (var id in selection) {
                if (selection[id]) {
                    selectedItems.push(this.grid.row(id).data);
                }
            }

            return selectedItems;
        },          

        getSortSpecs : function(params) {
            var pdf = params != null && params.pdf;

            var sortSpecs = [];
            for (var n = 0; this.grid.sort != null && n < this.grid.sort.length; n++) {
                var sort = this.grid.sort[n];
                if (pdf) {
                    var pdfProperty = this.getPdfColumnByField(sort.property);
                    if (pdfProperty != null) {
                        sort.property = pdfProperty;
                    }
                }
                sortSpecs.push({ first : sort.property, second : sort.descending });
            }

            return sortSpecs;
        },

        getColumnToWidth : function(params) {
	    var columnToWidth = new Object();
	    var columns = this.grid.get("columns");
	    for (var n = 0; n < columns.length; n++) {
		var column = columns[n];
		if (!column.hidden) {
		    columnToWidth[column.id] = (column.headerNode != null ? column.headerNode.clientWidth : null);
		}		    
	    }
            return columnToWidth;            
        },

        getActualHeight : function() {
            var height = 0;
            height += this.grid.headerNode.offsetHeight;

            var rowNodes = this.grid.contentNode.childNodes;
            for (var n = 0; n < rowNodes.length; n++) {
                height += rowNodes[n].offsetHeight;
            }

            height += 2;
            return height;            
        },

        getSortedData : function() {
            if (this.gridClass == "Grid") {
                var TrackableMemory = declare([FancyMemory, Trackable]);
                var store = new TrackableMemory({data : this.effectiveData, idProperty : this.idProperty});
                var dataPromise = store.filter({}).sort(this.grid.sort).fetch();
                return dataPromise;                              
            } else if (this.gridClass == "OnDemandGrid") {
                // NOTE: This function only works reliable, if a sort order actually is defined on
                //       the grid.  If data is loaded into the grid, without defining any sort
                //       order, the return value of this function will be unspecified, and probably
                //       not be related to the situation the user sees.              
                var dataPromise = this.store.filter({}).sort(this.grid.sort).fetch();
                return dataPromise;              
            } else {
                throw new Error("Unsupported gridClass: [" + this.gridClass + "]");
            }                
        },              

        destroy : function() {
            this.inherited(arguments);

            // Remove columnHider CSS rules, to avoid that they get active again when such a grid is constructed again at a later time.
            for (var id in this.columnHiderRules) {
                this.columnHiderRules[id].remove();
            }               

            if (this.grid != null) {
                this.grid.destroy();
            }
        }       
    });

    return ListWidget;
});
