/***********************************************************
* $Id$
*
* Copyright (C) 2017 ev-i Informationstechnologie Gmbh
*
**********************************************************/

define([ 
    "clazzes/TinyLog",
    "clazzes/canvas/CanvasHelper",
    "clazzes/canvas/image/TiledImage",
    "clazzes/mouse/LocalDragOperation",
    "clazzes/mouse/MouseHelper",
    "clazzes/widgets/layout/ContentWidget",
    "dojo/_base/declare",
    "dojo/_base/lang", 
    "dojo/dom-class",
    "dojo/dom-construct",
    "dojo/dom-style",
    "dojo/io-query",
    "dojo/on",
], function(
    TinyLog,
    CanvasHelper,
    TiledImage,
    LocalDragOperation,
    MouseHelper,
    ContentWidget,
    declare,
    lang,
    domClass,
    domConstruct,
    domStyle,
    ioQuery,
    on    
) {

    var className = "at.cdes.web.document.DocumentCompareCanvasWidget";

    var log = new TinyLog(className);
    var drawLog = new TinyLog(className + "_Draw");
    var imageObjLog = new TinyLog(className + "_ImageObj");    
    
    var DocumentCompareCanvasWidget = declare(className, ContentWidget, {

        constructor : function(params) {
            lang.mixin(this, params);

            this.canvasSteps = [
                { resolution :  36, x : 12, y : 54, width : 8, height : 6 },
                { resolution :  51, x : 24, y : 51, width : 8, height : 9 },
                { resolution :  72, x : 36, y : 48, width : 8, height : 12 },
                { resolution : 102, x : 48, y : 43, width : 8, height : 17 },
                { resolution : 144, x : 60, y : 36, width : 8, height : 24 },
                { resolution : 204, x : 72, y : 26, width : 8, height : 34 },
                { resolution : 288, x : 84, y : 12, width : 8, height : 48 }                
            ];

            // The region where the coordinates given above are located, including the plus/minus button below
            this.canvasButtonRegion = { x1 : 12, y1 : 12, x2 : 92, y2 : 77 };

            this.selectedCanvasStepIndex = 2;
            this.actionImageObjects = new Object();
            this.loadScaleModifierIcon("scaleMinus");
            this.loadScaleModifierIcon("scalePlus");

            this.contentDiv = domConstruct.create("div", null, null);

            this.imageCanvas = CanvasHelper.getMostNativeCanvasInstance();
            domClass.add(this.imageCanvas.domNode, "fixedDialogWidget documentCompareCanvasCanvas documentCompareCanvasImageCanvas");
            domConstruct.place(this.imageCanvas.domNode, this.contentDiv);

            this.toolCanvas = CanvasHelper.getMostNativeCanvasInstance();
            domClass.add(this.toolCanvas.domNode, "fixedDialogWidget documentCompareCanvasCanvas documentCompareToolCanvas");
            domConstruct.place(this.toolCanvas.domNode, this.contentDiv);

            this.idToImageObj = new Object();
            if (imageObjLog.isDebugEnabled()) {
                imageObjLog.debug("Constructor constructed new empty idToImageObj map.");
            }                

	    this.mouseDownHandler = {
		handleEvent : lang.hitch(this, this.mouseDownHandlerFunction)
	    };
	    this.contentDiv.addEventListener("mousedown", this.mouseDownHandler, false);

	    this.mouseMoveHandler = {
		handleEvent : lang.hitch(this, this.mouseMoveUpHandlerFunction)
	    };
	    this.contentDiv.addEventListener("mousemove", this.mouseMoveHandler, false);
	    this.mouseUpHandler = {
		handleEvent : lang.hitch(this, this.mouseMoveUpHandlerFunction)
	    };
	    this.contentDiv.addEventListener("mouseup", this.mouseUpHandler, false);
            
            this.mouseWheelHandler = {
                handleEvent : lang.hitch(this, this.mouseWheelHandlerFunction)
            };                
            this.contentDiv.addEventListener("mousewheel", this.mouseWheelHandler, false);

            domStyle.set(this.contentDiv, "cursor", "pointer");
        },

        getWidgetId : function() {
            return "DocumentCompareCanvasWidget";
        },

        getDataId : function() {
            return null;
        },

        getContainer : function() {
            return this.contentDiv;
        },

        resize : function(newSize) {
            this.width = newSize.w;
            this.height = newSize.h;            
            domStyle.set(this.contentDiv, "width", this.width + "px");
            domStyle.set(this.contentDiv, "height", this.height + "px");
            this.imageCanvas.resize(newSize);
            this.toolCanvas.resize(newSize);
        },

        getWidth : function() {
            return this.width;
        },

        getHeight : function() {
            return this.height;
        },

        setData : function(params) {
            lang.mixin(this, params);            

            // Forget about all cached images already loaded for a different DocumentVersionId
            this.idToImageObj = new Object();
            if (imageObjLog.isDebugEnabled()) {
                imageObjLog.debug("setData constructed new empty idToImageObj map");
            }                
            
            this.selectedCanvasStepIndex = 2;
            this.toolCanvasDirty = true;
            
            this.reload();            
        },

        reload : function() {
            
        },        

        registerTile : function(resolution, angle, x, y) {
            var ownDocumentVersionId = this.ownDocumentVersion.id;
            var chosenDocumentVersionId = this.chosenDocumentVersion.id;

            if (log.isDebugEnabled()) {
                log.debug("registerTile: docVersions = [" + ownDocumentVersionId + "], [" + chosenDocumentVersionId + "]");                
                log.debug(".... resolution = [" + resolution + "], angle = [" + angle + "], x = [" + x + "], y = [" + y + "]");
            }                
            
            var paramString = ioQuery.objectToQuery({
                service : "CDESMimeService/4/compareDocumentVersionPngTileMimeSource",                
                     sp : [ ownDocumentVersionId, chosenDocumentVersionId, this.tiledImage.getBaseSize(), "T", "d" + resolution, x, y, angle ],
                     ts : dojoConfig.tabSessionId              
            });
            
            var imageObject = new Image();
            imageObject.onload = lang.hitch(this, function() {
                this.imageLoadFailed = false;
                this.imageLoaded = true;
                this.repaint();
            });
            imageObject.onabort = lang.hitch(this, function() {
                this.imageLoadFailed = true;
                this.repaint();
            });
            imageObject.onerror = lang.hitch(this, function() {
                this.imageLoadFailed = true;
                this.repaint();
            });            
            imageObject.src = "/cdes/app?" + paramString;

            var tileId = TiledImage.getTileId(resolution, angle, x, y);
            this.idToImageObj[tileId] = imageObject;
            if (imageObjLog.isDebugEnabled()) {
                imageObjLog.debug("registerTile registers imageObj under tileId [" + tileId + "]; src = [" + imageObject.src + "]");
            }                
        },

        dropAllTiles : function(resolution, angle) {
            for (var tileId in this.idToImageObj) {
                if (tileId.indexOf(resolution + "_" + angle) == 0) {
                    delete this.idToImageObj[tileId];
                    if (imageObjLog.isDebugEnabled()) {
                        imageObjLog.debug("dropAllTiles deleted imageObj for key [" + tileId + "] from idToImageObj map");
                    }
                }                    
            }
        },

        dropTile : function(resolution, angle, x, y) {
            if (log.isDebugEnabled()) {
                log.debug("dropTile for resolution = [" + resolution + "], angle = [" + angle + "], x = [" + x + "], y = [" + y + "]");
            }                

            var tileId = TiledImage.getTileId(resolution, angle, x, y);
            delete this.idToImageObj[tileId];

            if (imageObjLog.isDebugEnabled()) {
                imageObjLog.debug("dropTile deleted imageObj for tileId [" + tileId + "] from idToImageObj map");
            }                
        },

        repaint : function() {
            if (drawLog.isDebugEnabled()) {
                drawLog.debug("Called repaint");
            }                

            var context = this.imageCanvas.getContext2D();
            context.clear();

            var tileInfos = this.tiledImage.getCurrentTileInfos();
            for (var tileY = 0; tileInfos != null && tileY < tileInfos.length; tileY++) {
                if (tileInfos[tileY] != null) {
                    for (var tileX = 0; tileX < tileInfos[tileY].length; tileX++) {
                        var tileInfo = tileInfos[tileY][tileX];
                        if (tileInfo != null) {
                            var resolution = this.tiledImage.getResolution();
                            var angle = this.tiledImage.getAngle();                            
                            var tileId = TiledImage.getTileId(resolution, angle, tileX, tileY);
                            var imageObject = this.idToImageObj[tileId];                            

                            if (drawLog.isDebugEnabled()) {
                                drawLog.debug("Draw: tileId = [" + tileId + "], (x,y)=(" + tileInfo.x + "," + tileInfo.y
                                    + "); (w,h)=(" + tileInfo.width + "," + tileInfo.height + ")");                                    
                            }                                

                            context.drawImage(imageObject, tileInfo.x, tileInfo.y, tileInfo.width, tileInfo.height);
                        }                            
                    }                        
                }                    
            }                

            this.drawTools();
        },

        drawTools : function() {
            // Avoid calling this more often than actually necessary.  In particular, when just moving the image,
            // repainting the tools is unnecessary.            
            if (this.toolCanvasDrity != null && !this.toolCanvasDirty) {
                return;
            }                
            
            var context = this.toolCanvas.getContext2D();
            context.clear();

            for (var z = 0; z < this.canvasSteps.length; z++) {
                var step = this.canvasSteps[z];
                this.drawScaleSetter(context, step.x, step.y, step.width, step.height, z == this.selectedCanvasStepIndex);
            }

            // CAUTION: Needs to be in sync with the coordinates in the mouseDownHandlerFunction
            this.drawScaleModifierIcon(context, 12, 64, 32, 13, "scaleMinus");
            this.drawScaleModifierIcon(context, 60, 64, 32, 13, "scalePlus");

            this.toolCanvasDirty = false;
        },

        drawScaleSetter : function(context, x, y, width, height, selected) {
            try {
                context.save();

                if (selected) {
                    context.setFillStyle("rgba(255, 0, 255, 1)");
                } else {
                    context.setFillStyle("rgba(0, 0, 0, 0.4)");
                }                    

                context.beginPath();
                context.rect(x, y, width, height);
                context.fill();                
            } finally {
                context.restore();
            }                
        },

        loadScaleModifierIcon : function(imageName) {
            if (this.actionImageObjects[imageName] == null) {
                var imageObj = new Image();
                var url = "/cdes/im/custom/" + imageName + ".png";       
                imageObj.src = url;

                imageObj.onload = lang.hitch(this, function() {
                    this.toolCanvasDirty = true;
                    this.drawTools();
                    if (log.isDebugEnabled()) {
                        log.debug("Image [" + url + "] successfully loaded.");
                    }                    
                });
                imageObj.onabort = lang.hitch(this, function() {
                    log.error("Loading image [" + url + "] was aborted.");
                });
                imageObj.onerror = lang.hitch(this, function() {
                    log.error("Loading image [" + url + "] failed");
                });
                this.actionImageObjects[imageName] = imageObj;                
            }
        },            

        drawScaleModifierIcon : function(context, x, y, width, height, imageName) {
            var imageObj = this.actionImageObjects[imageName];
            context.drawImage(imageObj, x, y, width, height);
        },

        getCanvasStepIndex : function(mouseX, mouseY) {
            for (var n = 0; n < this.canvasSteps.length; n++) {
                var canvasStep = this.canvasSteps[n];
                if (mouseX >= canvasStep.x && mouseX <= canvasStep.x + canvasStep.width && mouseY >= canvasStep.y && mouseY <= canvasStep.y + canvasStep.height) {
                    return n;
                }
            }

            return null;            
        },            

        isMouseOverMinusButton : function(mouseX, mouseY) {
            // CAUTION: Needs to be in sync with the coordinates in drawTools
            return mouseX >= 12 && mouseX <= 12 + 32 && mouseY >= 64 && mouseY <= 64 + 13;
        },

        isMouseOverPlusButton : function(mouseX, mouseY) {
            // CAUTION: Needs to be in sync with the coordinates in drawTools
            return mouseX >= 60 && mouseX <= 60 + 32 && mouseY >= 64 && mouseY <= 64 + 13;            
        },        

    	mouseDownHandlerFunction : function(mouseEvent) {
            if (MouseHelper.isLeftButtonPressed(mouseEvent)) {
                var mouseX = mouseEvent.offsetX - this.contentDiv.offsetLeft;
                var mouseY = mouseEvent.offsetY - this.contentDiv.offsetTop;
                var clickedCanvasStepIndex = this.getCanvasStepIndex(mouseX, mouseY);
                if (clickedCanvasStepIndex != null) {
                    this.setResolution(clickedCanvasStepIndex);

                    // If a resolution bar was clicked, processing is finished here.  In all other cases,
                    // we want to proceed to the drag handler below.                    
                    return;                    
                } else if (this.isMouseOverMinusButton(mouseX, mouseY)) {  
                    if (this.selectedCanvasStepIndex > 0) {
                        this.setResolution(this.selectedCanvasStepIndex - 1);
                    }
                    return;
                } else if (this.isMouseOverPlusButton(mouseX, mouseY)) {
                    if (this.selectedCanvasStepIndex < this.canvasSteps.length - 1) {
                        this.setResolution(this.selectedCanvasStepIndex + 1);
                    }            
                    return;
                }                    
            }                
            
    	    if (MouseHelper.isLeftButtonPressed(mouseEvent) || MouseHelper.isMiddleButtonPressed(mouseEvent)) {
        	mouseEvent.preventDefault();
        	mouseEvent.stopPropagation();
        	
        	this.dragOperation = new LocalDragOperation(mouseEvent, {
    		       domNode : this.contentDiv,
    	            dragFilter : function(mouseEvent) { return MouseHelper.isLeftButtonPressed(mouseEvent) || MouseHelper.isMiddleButtonPressed(mouseEvent); },
    	           performDrag : lang.hitch(this, this.performDrag),
    	           incremental : true,
            callPreventDefault : false
        	});	    			
            }
        },

        mouseMoveUpHandlerFunction : function(mouseEvent) {
            var mouseX = mouseEvent.offsetX - this.contentDiv.offsetLeft;
            var mouseY = mouseEvent.offsetY - this.contentDiv.offsetTop;

            var overButton = false;
            if (mouseX >= this.canvasButtonRegion.x1 && mouseX <= this.canvasButtonRegion.x2
                && mouseY >= this.canvasButtonRegion.y1 && mouseY <= this.canvasButtonRegion.y2) {

                    for (var n = 0; n < this.canvasSteps.length; n++) {
                        var canvasStep = this.canvasSteps[n];
                        overButton |= (mouseX >= canvasStep.x && mouseX <= canvasStep.x + canvasStep.width
                            && mouseY >= canvasStep.y && mouseY <= canvasStep.y + canvasStep.height);                        
                    }
                    overButton |= this.isMouseOverMinusButton(mouseX, mouseY);
                    overButton |= this.isMouseOverPlusButton(mouseX, mouseY);                    
                }

            var cursor = null;
            if (overButton) {
                cursor = "default";
            } else {
                if (this.dragOperation != null && this.dragOperation.isActive()) {
                    cursor = "all-scroll";
                } else {
                    cursor = "pointer";
                }                    
            }                

            if (log.isDebugEnabled()) {
                log.debug("Setting mouse cursor to [" + cursor + "]");
            }                

            domStyle.set(this.contentDiv, "cursor", cursor);
        },

        mouseWheelHandlerFunction : function(mouseEvent) {
            var wheelDelta = mouseEvent.wheelDelta;            
            log.info("Mouse wheel delta: ", wheelDelta);

            if (wheelDelta < 0) {
                if (this.selectedCanvasStepIndex > 0) {
                    this.setResolution(this.selectedCanvasStepIndex - 1);
                }
            } else if (wheelDelta > 0) {
                if (this.selectedCanvasStepIndex < this.canvasSteps.length - 1) {
                    this.setResolution(this.selectedCanvasStepIndex + 1);
                }            
            }                
        },            

        setResolution : function(clickedCanvasStepIndex) {
            var desiredResolution = this.canvasSteps[clickedCanvasStepIndex].resolution;
            if (log.isDebugEnabled()) {
                log.debug("Clicked canvas step [" + clickedCanvasStepIndex + "] with resolution [" + desiredResolution + "]");
            }

            this.selectedCanvasStepIndex = clickedCanvasStepIndex;
            this.toolCanvasDirty = true;
            this.tiledImage.setResolution(desiredResolution);            
        },

    	performDrag : function(mouseDx, mouseDy, mousePosition, event) {
    	    if (log.isDebugEnabled()) {
    		log.debug("performDrag: dx = " + mouseDx + ", dy = " + mouseDy);	
    	    }

            this.tiledImage.move(mouseDx, mouseDy);
        }
    });

    return DocumentCompareCanvasWidget;
});
