/* Copyright (C) 2022 ev-i Informationstechnologie GmbH */

define([
    "clazzes/LoginGuardRpcService",
    "clazzes/TinyLog",
    "clazzes/util/DOMHelper",
    "clazzes/util/ErrorHelper",
    "clazzes/util/JobHelper",
    "clazzes/util/WidgetHelper",
    "clazzes/util/StringHelper",
    "clazzes/widgets/layout/ContentWidget",
    "clazzes/widgets/layout/InfoDialog",
    "cdes/util/PasswordValidator",
    "cdes/util/SecurityQuestionValidator",
    "dijit/form/Button",
    "dijit/form/TextBox",
    "dojo/dom-class",
    "dojo/dom-construct",
    "dojo/dom-style",
    "dojo/io-query",
    "dojo/on",
    "dojo/_base/declare",
    "dojo/_base/lang",
    "dojo/string",
    //        "dojo/text!/cdes/svc/adminService?smd",
    //        "dojo/text!/cdes-dojo-impl/jobStatus?smd",
    "dojo/i18n!/cdes/nls/cdes-web-i18n.js"
],
       function(
    	   LoginGuardRpcService,
    	   TinyLog,
    	   DOMHelper,
           ErrorHelper,
           JobHelper,
           WidgetHelper,
           StringHelper,
    	   ContentWidget,
    	   InfoDialog,
           PasswordValidator,
           SecurityQuestionValidator,
    	   Button,
	   	   TextBox,
    	   domClass,
    	   domConstruct,
    	   domStyle,
           ioQuery,
    	   on,
    	   declare,
    	   lang,
           string,
	   //    		 adminServiceSmd,
	   //    		 jobStatusServiceSmd,
    	   i18n
       ) {

	   var className = "at.cdes.web.challengeLogin.NewPasswordPage";

	   var log = new TinyLog(className);

	   var NewPasswordPage = declare(className, ContentWidget, {

    	   constructor : function(params) {
    		   lang.mixin(this,params);

    		   this.topDiv = this.constructTopDiv();

    		   this.allFieldsValid = true;
               this.lastResult = undefined;

           	   this.setData();
    	   },

    	   getWidgetId : function() {
    		   return "NewPasswordPage";
   	   },

    	   getDataId : function() {
    		   return null;
    	   },

    	   getContainer : function() {
    		   return this.topDiv;
    	   },

    	   constructTopDiv : function() {
    		   var topDiv = domConstruct.create("div", null, null);
               topDiv.classList.add("newPasswordPageTopDiv");

	           // IntroText
	           this.introText = DOMHelper.createTextNode("h4", "", topDiv, null);

			   this.inputsDiv = domConstruct.create("div", null, null);
               domConstruct.place(this.inputsDiv, topDiv);

               this.oldPasswordTextBox = new TextBox({
                   intermediateChanges: true,
                   onChange: lang.hitch(this, function() {
                       if (this.lastResult === "WRONG_PASSWORD") {
                           this.lastResult = undefined;
                       }
                       this.updateWidgetState();
                   }),
                   type: "password",
                   autocomplete: "current-password",
                   label: i18n.newPasswordOldPassword,
                   "class": "newPasswordCol2 newPasswordRow1",
               });
               // https://bugs.dojotoolkit.org/ticket/9562
               // dijit.TextBox does not simply support autocomplete
               this.oldPasswordTextBox.set("autocomplete", "current-password");
               domConstruct.place(this.oldPasswordTextBox.domNode, this.inputsDiv);
		       DOMHelper.createTextNode("div", i18n.newPasswordOldPassword, this.inputsDiv, "newPasswordCol1 newPasswordRow1");

               this.newPassword1TextBox = new TextBox({
                   intermediateChanges: true,
                   onChange: lang.hitch(this, function() {
                       if (this.lastResult === "INVALID_PASSWORD") {
                           this.lastResult = undefined;
                       }
                       this.updateWidgetState();
                   }),
                   type: "password",
                   label: i18n.newPasswordNewPassword,
                   "class": "newPasswordCol2 newPasswordRow2",
               });
               this.newPassword1TextBox.set("autocomplete", "new-password");
               domConstruct.place(this.newPassword1TextBox.domNode, this.inputsDiv);
		       DOMHelper.createTextNode("div", i18n.newPasswordNewPassword, this.inputsDiv, "newPasswordCol1 newPasswordRow2");

               this.newPassword2TextBox = new TextBox({
                   intermediateChanges: true,
                   onChange: lang.hitch(this, this.updateWidgetState),
                   type: "password",
                   label: i18n.newPasswordNewPassword2,
                   "class": "newPasswordCol2 newPasswordRow3",
               });
               this.newPassword2TextBox.set("autocomplete", "new-password");
               domConstruct.place(this.newPassword2TextBox.domNode, this.inputsDiv);
		       DOMHelper.createTextNode("div", i18n.newPasswordNewPassword2, this.inputsDiv, "newPasswordCol1 newPasswordRow3");

               this.securityQuestionDiv = domConstruct.create("div");
               this.securityQuestionDiv.style.display = "none";

               this.questionTextBox = new TextBox({
                   intermediateChanges: true,
                   onChange: lang.hitch(this, this.updateWidgetState),
                   label: i18n.challengeLoginPasswordQuestionLabel,
                   "class": "newPasswordCol2 newPasswordRow4",
               });
               domConstruct.place(this.questionTextBox.domNode, this.securityQuestionDiv);
		       DOMHelper.createTextNode("div", i18n.challengeLoginPasswordQuestionLabel, this.securityQuestionDiv, "newPasswordCol1 newPasswordRow4");

               this.answerTextBox = new TextBox({
                   intermediateChanges: true,
                   onChange: lang.hitch(this, this.updateWidgetState),
                   label: i18n.challengeLoginPasswordAnswerLabel,
                   "class": "newPasswordCol2 newPasswordRow5",
               });
               domConstruct.place(this.answerTextBox.domNode, this.securityQuestionDiv);
		       DOMHelper.createTextNode("div", i18n.challengeLoginPasswordAnswerLabel, this.securityQuestionDiv, "newPasswordCol1 newPasswordRow5");
               domConstruct.place(this.securityQuestionDiv, this.inputsDiv);

               this.applicationContext.getService("resetPasswordService").getResetPasswordInfo().then(lang.hitch(this, function (info) {
                       this.needsSecurityQuestion = info.needsSecurityQuestion;
                       this.passwordMaxAge = info.passwordMaxAge;
                       this.passwordRemainingDays = info.passwordRemainingDays;
                       this.securityAnswerMinLength = info.securityAnswerMinLength;
                       this.passwordMinLength = info.passwordMinLength;
                       this.updateWidgetState();
                   }));

		       this.hintLine = domConstruct.create("div", null, null);
		       domClass.add(this.hintLine, "fixedDialogWidget challengeLoginHintLine invisible newPasswordCol1 newPasswordRow7");
		       domConstruct.place(this.hintLine, this.inputsDiv);

		       this.hintIcon = domConstruct.create("div", null, this.hintLine);
	    	   domClass.add(this.hintIcon, "challengeLoginHintIcon clazzesWarnIcon32");

	    	   // Hint Div
	    	   this.hintSpan = DOMHelper.createTextNode("span", "", this.hintLine, "challengeLoginHintSpan");

               const buttonDiv = domConstruct.create("div", {
                   "class": "newPasswordCol1 newPasswordRow6",
               }, this.inputsDiv);
               this.submitButton = new Button({
                   onClick: lang.hitch(this, this.resetPassword),
                   label: i18n.newPasswordSubmitLabel,
               });
		       domConstruct.place(this.submitButton.domNode, buttonDiv);

               this.closeButton = new Button({
                   onClick: lang.hitch(this, function() {
                       document.dispatchEvent(new CustomEvent("close"));
                   }),
                   label: i18n.closeButtonCaption,
               });
		       domConstruct.place(this.closeButton.domNode, buttonDiv);

               this.successDiv = domConstruct.create("div", { style: { display: "none" } }, null);
               domConstruct.place(this.successDiv, topDiv);

    		   return topDiv;
    	   },

           resetPassword: function() {
               var oldPassword = this.oldPasswordTextBox.get("value");
               var newPassword = this.newPassword1TextBox.get("value");
               var question = this.questionTextBox.get("value");
               var answer = this.answerTextBox.get("value");

			   var organisationPersonId = this.applicationContext.getPageContextOrganisationPersonId();
               this.applicationContext.getService("resetPasswordService")
                   .resetPassword(organisationPersonId, oldPassword, newPassword, question || undefined, answer || undefined)
                   .then(lang.hitch(this, function (result) {
                       this.lastResult = result;
                       this.updateWidgetState();
                       if (result == "SUCCESS") {
                           document.dispatchEvent(new CustomEvent("success"));
                       } else {
                           document.dispatchEvent(new CustomEvent("failure"));
                       }
                   }), lang.hitch(this, function (error) {
                       this.lastResult = "ERROR";
                       this.updateWidgetState();
                       document.dispatchEvent(new CustomEvent("failure"));
                }));
           },

    	   updateWidgetState : function() {
               var username = this.username;
               var oldPassword = this.oldPasswordTextBox.get("value");
               var newPassword1 = this.newPassword1TextBox.get("value");
               var newPassword2 = this.newPassword2TextBox.get("value");
               var question = this.questionTextBox.get("value");
               var answer = this.answerTextBox.get("value");

               this.securityQuestionDiv.style.display = this.needsSecurityQuestion ? "" : "none";

               if (newPassword1 !== this.lastPassword1 && !(newPassword2 == null || newPassword2.length === 0) && newPassword1 == newPassword2) {
                   this.lastServerValidationResult = undefined;

                   clearTimeout(this.serverValidationTimeout);
                   this.serverValidationTimeout = setTimeout(lang.hitch(this, function() {
                       this.applicationContext.getService("resetPasswordService")
                           .validatePassword(newPassword1)
                           .then(lang.hitch(this, function(result) {
                               this.lastServerValidationResult = result;
                               this.updateWidgetState();
                           }));
                       this.serverValidationTimeout = undefined;
                   }), 300);

               }

               this.lastPassword1 = newPassword1;

               var passwordValidationError = PasswordValidator.validate(newPassword1, this.passwordMinLength);
               var answerValidationError = SecurityQuestionValidator.validateAnswer(answer, this.securityAnswerMinLength);

               var hintButValid = false;
               var hintText = null;
			   var passwordTooShortWarning = string.substitute(i18n.challengeLoginPasswordTooShortWarning, {
	                		 passwordMinLength : !StringHelper.isEmpty(this.passwordMinLength) ? this.passwordMinLength : "",
	           });
               if (this.lastResult === "WRONG_PASSWORD") {
                   hintText = i18n.newPasswordWrongPassword;
               } else if (this.lastResult === "INVALID_PASSWORD") {
                   // because we disable the submit button if the user enters an invalid password this should never happen.
                   log.warn("Inconsistency between server and client side checking of password.");
                   hintText = i18n.newPasswordInvalidPassword;
               } else if (this.lastResult === "ERROR") {
                   hintButValid = true;
                   hintText = i18n.newPasswordError;
               } else if (this.lastResult === "SUCCESS") {
                   hintButValid = true;
                   hintText = i18n.newPasswordSuccess;
               } else if (oldPassword == null || oldPassword.length === 0) {
                   hintText = i18n.newPasswordNoOldPassword;
               } else if (newPassword1 == null || newPassword1.length === 0) {
                   hintText = i18n.newPasswordNoNewPassword;
               } else if ((passwordValidationError === "MISSING" && this.lastServerValidationResult == null) || this.lastServerValidationResult === "MISSING") {
                   hintText = i18n.newPasswordNoNewPassword;
               } else if ((passwordValidationError === "TOO_SHORT" && this.lastServerValidationResult == null) || this.lastServerValidationResult === "TOO_SHORT") {
	        	   hintText = passwordTooShortWarning;
               } else if ((passwordValidationError === "LACKING_DIVERSITY" && this.lastServerValidationResult == null) || this.lastServerValidationResult === "LACKING_DIVERSITY") {
	           hintText = i18n.challengeLoginPasswordLackingDiversityWarning;
               } else if ((passwordValidationError === "ILLEGAL_CHARS" && this.lastServerValidationResult == null) || this.lastServerValidationResult === "ILLEGAL_CHARS") {
                   hintText = i18n.passwordIllegalCharsWarning;
               } else if (this.lastServerValidationResult === "OLD_PW") {
                   hintText = i18n.challengeLoginPasswordOldPwWarning;
               } else if (newPassword2 == null || newPassword2.length === 0) {
                   hintText = i18n.newPasswordNoRepeatPassword;
               } else if (newPassword1 !== newPassword2) {
                   hintText = i18n.newPasswordPasswordMismatch;
               } else if (question == null || question.length === 0 && this.needsSecurityQuestion) {
                   hintText = i18n.challengeLoginNoQuestionWarning;
               } else if (answerValidationError !== "OK" && this.needsSecurityQuestion) {
                   hintText = SecurityQuestionValidator.getAnswerMessage(answerValidationError, this.securityAnswerMinLength);
               } else  {
                   hintButValid = true;
                   hintText = i18n.newPasswordAdditionalInfoText;
               }

			   var introText = string.substitute(i18n.newPasswordIntroText, {
	                		 passwordMaxAge : !StringHelper.isEmpty(this.passwordMaxAge) ? this.passwordMaxAge : "",
	                  passwordRemainingDays : !StringHelper.isEmpty(this.passwordRemainingDays) ? this.passwordRemainingDays : ""
	           });
			   DOMHelper.setInnerText(this.introText, introText);

			   DOMHelper.setInnerText(this.hintSpan, hintText);

               var allFieldsValid = hintText == null || hintButValid;

               this.hintLine.classList.toggle("visible", hintText != null);
               this.hintLine.classList.toggle("invisible", hintText == null);

               this.submitButton.set("disabled", !allFieldsValid);

               var wasSuccessful = this.lastResult === "OK";

               this.inputsDiv.style.display = wasSuccessful ? "none" : "";
               this.successDiv.style.display = wasSuccessful ? "" : "none";
    	   },

    	   resize : function(newSize) {
		       // Account for margin = 8 of body
		       domStyle.set(this.topDiv, "width", (newSize.w - 16) + "px");
		       domStyle.set(this.topDiv, "height", (newSize.h - 16 - 4) + "px");
	       }
    });

	   return NewPasswordPage;
});
