/***********************************************************
 * $Id$
 *
 * Copyright (C) 2017 ev-i Informationstechnologie Gmbh
 *
 **********************************************************/

define([ "clazzes/TinyLog",
	"clazzes/util/ErrorHelper",
	"dijit/Destroyable",
	"dojo/_base/declare",
	"dojo/_base/lang",
	"dojo/i18n!/cdes/nls/cdes-web-i18n.js"],
	function(TinyLog,
			ErrorHelper,
			Destroyable,
			declare,
			lang,
			i18n) {

	var className = "at.cdes.web.planning.notification.SharepointTreeModel";

	var log = new TinyLog(className);

	var SharepointTreeModel = declare(className, Destroyable, {
		constructor : function(params) {
			lang.mixin(this, params);

			this.idToNode = { root : { id : "root", label: 'root', type: 'root', path: 'root'} };
			this.idToRunningDeferred = {};
		},

		destroy: function() {
			this.idToNode = { root : { id : "root", label: 'root', type: 'root', path: 'root'} };
		},


		// API methods:
		getRoot: function(onItem){
			// summary:
			//		Calls onItem with the root item for the tree, possibly a fabricated item.
			//		Throws exception on error.
			// tags:
			//		extension
			onItem(this.idToNode.root);
		},

		mayHaveChildren: function(node) {
			return node.type == "PROJECT" || node.type == "FOLDER";
		},

		getChildren: function(parentItem,onComplete,onError){
			var mspService = this.applicationContext.getService("mspService");

			if (parentItem.path == "root") {
				if (lang.isArray(parentItem.items)) {
					onComplete(parentItem.items);
				} else {
					if (!this.getRootDeferred ) {
						if (log.isDebugEnabled()) {
							log.debug("Calling getProjects()");
						}

						this.getRootDeferred = mspService.getProjects();
					}

					this.getRootDeferred.then(
							lang.hitch(this,function(data){
								delete this.getRootDeferred;
								this.putNodes(parentItem, data);
								onComplete(parentItem.items);
							}),
							lang.hitch(this,function(err){
								delete this.getRootDeferred;

								ErrorHelper.processAsyncError({
									err : err,
									widget : this,
									asyncOperation : null,
									opName : "getProjects()",
									message : i18n.sharePointTreeModelGetProjectsFailed
								});						    	 
								onError(err);
							}))
							.otherwise(lang.hitch(this,function(err){
								log.error("Handler of getProjects failed", err);
							}));
				}
			} else if (parentItem.type == "PROJECT") {
				var id = parentItem.id;
				if (lang.isArray(parentItem.items)) {
					onComplete(parentItem.items);
				} else {
					if (!(id in this.idToRunningDeferred)) {
						if (log.isDebugEnabled()) {
							log.debug("Calling getProjects()");
						}

						this.idToRunningDeferred[id] = mspService.getProjectFolders(parentItem.path);
					}

					this.idToRunningDeferred[id].then(
							lang.hitch(this,function(children){
								delete this.idToRunningDeferred[id];

								this.buildFolderTree(children);
								var directChildItems = [];
								for (var n = 0; n < children.length; n++) {
									if (children[n].positions != null && children[n].positions.length == 1) {
										directChildItems.push(children[n]);
									}
								}				

								this.putNodes(parentItem, directChildItems);
								onComplete(parentItem.items);
							}),
							lang.hitch(this,function(err){
								delete this.idToRunningDeferred[id];

								ErrorHelper.processAsyncError({
									err : err,
									widget : this,
									asyncOperation : null,
									opName : "getProjectFolders",
									message : i18n.sharePointTreeModelGetProjectFoldersFailed
								});						    	 
								onError(err);
							}))
							.otherwise(lang.hitch(this,function(err){
								log.error("Handler of getProjectFolders failed", err);
							}));
				}
			} else if (parentItem.type == "FOLDER") {
				this.putNodes(parentItem, parentItem.children);
				onComplete(parentItem.items);
			} else {
				throw new Error("Node type [" + parentItem.type + "] not supported by getChildren.");
			}		
		},

		putNodes: function(parentItem, children) {

			parentItem.items = [].concat(children);

			for (var i = 0; i < parentItem.items.length; i++) {
				var node = parentItem.items[i];
				this.putNode(node);
			}
		},

		putNode: function(node) {
			var originalNode = this.idToNode[node.id];

			if (originalNode) {
				if (log.isDebugEnabled()) {
					log.debug("Mixin node with id [" + node.id + "] to previously cached node.");
				}

				lang.mixin(originalNode, node);
				return originalNode;
			} else {
				this.idToNode[node.id] = node;
				return node;
			}
		},

		isItem: function(item){
			return item && item.id !== undefined && this.idToNode[item.id] !== undefined;
		},		

		fetchItemByIdentity: function(keywordArgs){
			// summary:
			//		Given the identity of an item, this method returns the item that has
			//		that identity through the onItem callback.  Conforming implementations
			//		should return null if there is no item with the given identity.
			//		Implementations of fetchItemByIdentity() may sometimes return an item
			//		from a local cache and may sometimes fetch an item from a remote server.
			// tags:
			//		extension

			var item = this.idToNode[keywordArgs.identity];

			if (item !== undefined) {
				lang.hitch(keywordArgs.scope, keywordArgs.onItem)(item);
			}
		},

		buildFolderTree : function(folders) {
			for (var n = 0; n < folders.length; n++) {
				var title = folders[n].title;

				var positions = [];
				if (title != null) {
					var currIndex = 0;
					while (currIndex >= 0) {
						var nextDotPosition = title.indexOf('.', currIndex);
						if (nextDotPosition != -1) {
							var numberString = title.substring(currIndex, nextDotPosition);
							var numberInt = parseInt(numberString);
							if (numberInt != null && !isNaN(numberInt) && numberInt >= 0) {
								positions.push(numberInt);				
								currIndex = nextDotPosition + 1;
							} else {
								currIndex = -1;
							}
						} else {
							currIndex = -1;
						}			    
					}		
				}

				folders[n].positions = positions;

				var positionKey = "";
				var parentPositionKey = null;
				for (var z = 0; z < positions.length; z++) {
					positionKey += positions[z];

					if (z == positions.length - 2) {
						parentPositionKey = positionKey;
					}

					if (z < positions.length - 1) {
						positionKey += ".";
					}
				}

				folders[n].positionKey = positionKey;
				folders[n].parentPositionKey = parentPositionKey;		
				folders[n].children = [];
			}		

			var positionKeyToFolder = new Object();
			var currLevel = 0;
			var progress = true;
			while (progress) {
				var currProgress = false;
				for (var n = 0; n < folders.length; n++) {
					var folder = folders[n];
					if (folder.positions.length == currLevel) {
						if (folder.parentPositionKey != null) {
							// The usual key, just take the parent from the map
							var parentFolder = positionKeyToFolder[folder.parentPositionKey];

							// Unfortunately, in some cases the intermediate folder is missing.
							// Example: Folder 7. exists, folder 7.1. does not exist, but folder 7.1.6. does exist again.
							// Then traverse the tree upwards to the root, and use the first upper folder instead.			    
							var parentLevel = currLevel - 2;
							while (parentFolder == null && parentLevel >= 0) {
								var grandparentKey = "";
								for (var v = 0; v < folder.positions.length && v <= parentLevel; v++) {
									grandparentKey += folder.positions[v];
									if (v < parentLevel) {
										grandparentKey += ".";
									}

									if (grandparentKey in positionKeyToFolder) {
										parentFolder = positionKeyToFolder[grandparentKey];
									}

									parentLevel--;				    
								}
							}

							// If there is no upper folder as well, then we cannot do anything senseful here.
							if (parentFolder != null) {
								parentFolder.children.push(folder);
							}
						}

						positionKeyToFolder[folder.positionKey] = folder;
						currProgress = true;			
					}
				}

				currLevel++;
				progress = currProgress;		
			}

			for (var n = 0; n < folders.length; n++) {
				folders[n].children.sort(function(folderOne, folderTwo) {
					return folderOne.positionKey.localeCompare(folderTwo.positionKey);
				});		    
			}		
		},	    

		getIdentity: function(item){
			// summary:
			//		Returns identity for an item
			// tags:
			//		extension

			if (item && item.id !== undefined && this.idToNode[item.id] !== undefined) {
				return item.id;
			} else {
				return null;
			}
		},

		getNodeById : function(id) {
			return (id in this.idToNode ? this.idToNode[id] : null); 
		},

		getTooltip: function(item){
			return item.title;
		},

		getLabel: function(item) {
			return item.title;  
		},

		getItemClass: function(item) {
			return null;
		}
	});

	return SharepointTreeModel;
});




/* Code for loading tree nodes with child folders and documents, probably not needed.
		var id = parentItem.id;
		if (lang.isArray(parentItem.items)) {
		    onComplete(parentItem.items);
		} else {
		    if (!(id in this.idToRunningDeferred)) {
			if (log.isDebugEnabled()) {
			    log.debug("Calling getProjects()");
			}		

			this.idToRunningDeferred[id] = mspService.getFolderDocuments(parentItem.path, parentItem.guid, this.sharepointReferenceNames);
		    }

		    this.idToRunningDeferred[id].then(
			lang.hitch(this,function(documents){
			    delete this.idToRunningDeferred[id];

			    var allChildren = [];
			    for (var n = 0; n < parentItem.children.length; n++) {
				allChildren.push(parentItem.children[n]);
			    }				
			    for (var n = 0; n < documents.length; n++) {
				allChildren.push(documents[n]);
			    }				

			    this.putNodes(parentItem, allChildren);
			    onComplete(parentItem.items);
			}),
			lang.hitch(this,function(err){
			    delete this.idToRunningDeferred[id];

			    ErrorHelper.processAsyncError({
					   err : err,
					widget : this,
				asyncOperation : null,
		        		opName : "getProjectFolders",
		        	       message : i18n.sharePointTreeModelGetFolderDocumentsFailed
			    });						    	 
			    onError(err);
			}))
			.otherwise(lang.hitch(this,function(err){
			    log.error("Handler of getProjectFolders failed", err);
			}));
		}		*/
