/***********************************************************
* $Id$
*
* Copyright (C) 2017 ev-i Informationstechnologie Gmbh
*
**********************************************************/

//// css-prefix = organisationList
//// i18n-prefix = organisationList


define(["cdes/core/CdesVoc",
        "cdes/organisation/OrganisationHelper",
        "cdes/util/DialogHelper",
        "cdes/util/NameHelper",
        "cdes/widget/base/ListWidget",
        "clazzes/TinyLog",
        "clazzes/form/FancyButton",
        "clazzes/topic",
        "clazzes/util/DOMHelper",
        "clazzes/util/DataHelper",
        "clazzes/util/ErrorHelper",
        "clazzes/util/StringHelper",
        "clazzes/widgets/layout/ContentWidget",
        "clazzes/widgets/layout/InfoDialog",
        "dgrid/OnDemandGrid",
        "dgrid/extensions/ColumnResizer",
        "dgrid/extensions/DijitRegistry",
        "dojo/aspect",
        "dojo/dom-class",
        "dojo/dom-construct",
        "dojo/on",
        "dojo/string",
        "dojo/_base/declare",
        "dojo/_base/lang",
        "dojo/i18n!/cdes/nls/cdes-web-i18n.js"],
function(CdesVoc,
         OrganisationHelper,
         DialogHelper,
         NameHelper,
         ListWidget,
         TinyLog,
         FancyButton,
         topic,
         DOMHelper,
         DataHelper,
         ErrorHelper,
         StringHelper,
         ContentWidget,
         InfoDialog,
         OnDemandGrid,
         ColumnResizer,
         DijitRegistry,
         aspect,
         domClass,
         domConstruct,
         on,
         string,
         declare,
         lang,
         i18n) {

    var className = "at.cdes.web.organisation.OrganisationListWidget";

    var log = new TinyLog(className);

    var OrganisationListWidget = declare(className, ListWidget, {

        dataAttributeName : "organisationJoins",

        constructor : function(params) {
            lang.mixin(this, params);

            this.topDiv = this.constructTopDiv();

            this.columnFieldToPdfColumn = {
                _organisationString : "name",
                _addressString : "address",
                _phoneString : "phone",
                _networksString : "networks"
            };

            this.allFieldsValid = true;
        },

        getWidgetId : function() {
            return "OrganisationListWidget";
        },

        getDataId : function() {
            return null;
        },

        getContainer : function() {
            return this.topDiv;
        },

        columnWidthKey : "organisationList/columnWidths",
        defaultColumnWidth : 130,

        columnIdToDefaultWidth : {
            /* TODO
            task : 130,
            serialNumber : 70,
            text : 350,
            planningNotification : 200,
            baulos : 130,
            project : 120,
            subjectArea : 130,
            authorizedInspector : 160,
            reference : 120,
            statutoryDuty : 130,
            documentPosition : 200,
            documentPage : 80,
            personInCharge : 280,
            dueDate : 120,
            state : 120,            
            actions : 110,
            selection : 90            
            */
        },            

        getLocalStorageContextKeys : function() {
            var organisationPersonId = this.applicationContext.getPageContextOrganisationPersonId();
            return [organisationPersonId];
        },        

        constructColumns : function() {
            var columns = [];

            columns.push({
                   field : "_organisationString",
                      id : "name",
                sortable : true,
                   label : i18n.organisationLabel
            });

            if (!this.similarOrganisationMode) {
                columns.push({
                        field : "_addressString",
                           id : "address",
                    formatter : OrganisationHelper.organisationAddressFormatter,
                     sortable : true,
                        label : i18n.organisationListAddressColumnCaption
                });            

                columns.push({
                         field : "_phoneString",
                            id : "phone",
                    renderCell : lang.hitch(this, this.renderPhone),
                      sortable : true,
                         label : i18n.organisationListPhoneColumnCaption
                });            
            }

            columns.push({
                     field : "_networksString",
                        id : "networks",
                renderCell : lang.hitch(this, this.renderNetworks),
                  sortable : true,
                     label : i18n.organisationListNetwoksColumnCaption
            });            

            columns.push({
                     field : "id",
                        id : "actions",
                renderCell : lang.hitch(this, this.renderActionCell),
                  sortable : false,
                     label : i18n.actionColumnCaption
            });

            return columns;
        },

        getPdfColumnByField : function(field) {
            return this.columnFieldToPdfColumn[field];
        },

        renderPhone : function(organisationJoin, data, cell) {
            var mail = StringHelper.getNonNullString(organisationJoin.organisationEmailAddress);
            var phone = StringHelper.getNonNullString(organisationJoin.organisationTelephoneNumber);
            var mobile = StringHelper.getNonNullString(organisationJoin.organisationMobileTelephoneNumber);
            var fax = StringHelper.getNonNullString(organisationJoin.organisationFacsimileTelephoneNumber);

            var phoneTable = OrganisationHelper.constructOrganisationPersonTable({
                  mail : mail,
                 phone : phone,
                mobile : mobile,
                   fax : fax
            });
            return phoneTable;
        },

        getPhoneString : function(organisationJoin) {
            var mail = StringHelper.getNonNullString(organisationJoin.organisationEmailAddress).trim();
            var phone = StringHelper.getNonNullString(organisationJoin.organisationTelephoneNumber).trim();
            var mobile = StringHelper.getNonNullString(organisationJoin.organisationMobileTelephoneNumber).trim();
            var fax = StringHelper.getNonNullString(organisationJoin.organisationFacsimileTelephoneNumber).trim();
            return mail + " " + phone + " " + mobile + " " + fax;
        },
        
        renderNetworks : function(organisationJoin, data, cell) {
            var networks = organisationJoin.networks;
            var parentDiv = domConstruct.create("div", null, null);
            for (var n = 0; n < networks.length; n++) {
                var network = networks[n];
                //var networkName = memberInfo.organisationJoin.organisationName;
                //var cssClass = OrganisationHelper.hasOrganisationInfoContextNetwork(this.applicationContext, memberInfo) ? "" : "consortiumListForeignNetworkMember";
                DOMHelper.createTextNode("div", network.name, parentDiv, "");
            }
            return parentDiv;
        },

        getNetworksString : function(organisationJoin) {
            var networksString = "";            
            var networks = organisationJoin.networks;
            for (var n = 0; n < networks.length; n++) {
                var network = networks[n];
                networksString += (n == 0 ? "" : " ") + network.name;
            }

            networksString = networksString.trim();

            return networksString;
        },

        renderActionCell : function(organisationJoin, data, cell) {
            var buttonDiv = domConstruct.create("div", null, null);

            var contextNetworkId = this.applicationContext.getPageContextNetworkId();
            if (this.mode == ContentWidget.Mode.SHOW && "showOrganisation" in organisationJoin.actions){
                var showButton = this.constructShowButton(organisationJoin);
                domConstruct.place(showButton.domNode, buttonDiv);
            } else {
	            if (organisationJoin._foreignNetwork /*&& referenceOrganisation" in organisationJoin.actions*/) {
	                var referenceButton = this.constructReferenceButton(organisationJoin);
	                domConstruct.place(referenceButton.domNode, buttonDiv);
	            }
	            if ("deleteOrganisation" in organisationJoin.actions && !this.consortiumMemberMode) {
		            var deleteButton = this.constructDeleteButton(organisationJoin);
	                domConstruct.place(deleteButton.domNode, buttonDiv);
	            }
	            if ("editOrganisation" in organisationJoin.actions) {
	                if (this.consortiumMemberMode) {
		                var removeButton = this.constructRemoveButton(organisationJoin);
		                domConstruct.place(removeButton.domNode, buttonDiv);
		            } else {
		            	var editButton = this.constructEditButton(organisationJoin);
	                	domConstruct.place(editButton.domNode, buttonDiv);
		            }
	            } else if ("showOrganisation" in organisationJoin.actions) {
	                var showButton = this.constructShowButton(organisationJoin);
	                domConstruct.place(showButton.domNode, buttonDiv);
	            }
            }
            
            return buttonDiv;
        },

        constructRemoveButton : function(organisationJoin) {
            var button = new FancyButton({
                    title : i18n.organisationListRemoveToolTip,
                iconClass : "fancyButtonIcon17x18 fancyButton17x18 deleteButton"
            });

            domClass.add(button.domNode, "listButton");

            on(button, "click", lang.hitch(this, function(event) {
                this.showRemoveQuestion(organisationJoin);
            }));

            return button;
        },

        constructDeleteButton : function(organisationJoin) {
            var button = new FancyButton({
                    title : i18n.organisationListDeleteToolTip,
                iconClass : "fancyButtonIcon17x18 fancyButton17x18 deleteButton"
            });

            domClass.add(button.domNode, "listButton");

            on(button, "click", lang.hitch(this, function(event) {
                this.prepareDeleteOrganisation(organisationJoin);
            }));

            return button;
        },

        constructEditButton : function(organisationJoin) {
            var button = new FancyButton({
                    title : i18n.organisationListEditToolTip,
                iconClass : "fancyButtonIcon17x18 fancyButton17x18 editButton"
            });

            domClass.add(button.domNode, "listButton");

            on(button, "click", lang.hitch(this, function() {
                this.applicationContext.setPage("organisationEdit",
                                                { mode : ContentWidget.Mode.EDIT, id : organisationJoin.organisationId },
                                                null, this.backPage, this.backParams);
            }));
            return button;
        },

        constructReferenceButton : function(organisationJoin) {
            var button = new FancyButton({
                    title : i18n.organisationListReferenceToolTip,
                iconClass : "fancyButtonIcon17x18 fancyButton17x18 referenceButton"
            });

            domClass.add(button.domNode, "listButton");

            on(button, "click", lang.hitch(this, function(event) {
                OrganisationHelper.askAddOrganisationToAdminNetwork({
                       applicationContext : this.applicationContext,
                    administratedNetworks : organisationJoin._notYetReferencedOrgs,
                         organisationJoin : organisationJoin,
                                   widget : this
                });
            }));

            return button;
        },

        constructShowButton : function(organisationJoin) {
            var button = new FancyButton({
                    title : i18n.organisationListShowToolTip,
                iconClass : "fancyButtonIcon17x18 fancyButton17x18 showButton"
            });

            domClass.add(button.domNode, "listButton");

            on(button, "click", lang.hitch(this, function() {
                this.applicationContext.setPage("organisationEdit",
                                                { mode : ContentWidget.Mode.SHOW, id : organisationJoin.organisationId },
                                                null, this.backPage, this.backParams);
            }));
            return button;
        },

        prepareDeleteOrganisation : function(organisationJoin) {
            var organisationPersonId = this.applicationContext.getPageContextOrganisationPersonId();

            this.registerAsyncOperationStarted(OrganisationListWidget.AsyncOperation.GET_DELETE_INFO);
            var organisationService = this.applicationContext.getService("organisationService");
            organisationService.getOrganisationInUseHints(organisationPersonId, organisationJoin.organisationId).then(
                lang.hitch(this, function(hints) {
                    this.registerAsyncOperationFinished(OrganisationListWidget.AsyncOperation.GET_DELETE_INFO);

                    if (hints.length == 0) {
                        this.showDeleteQuestion(organisationJoin);
                    } else {
                        this.showNotDeletableDialog(organisationJoin, hints);
                    }
                }), lang.hitch(this, function(err) {
                    ErrorHelper.processAsyncError({
                                   err : err,
                                widget : this,
                        asyncOperation : OrganisationListWidget.AsyncOperation.GET_DELETE_INFO,
                                opName : "getOrganisationInUseHints",
                               message : i18n.organisationListWidgetGetOrganisationInUseHintsFailed
                    });
                })).otherwise(
                    lang.hitch(this, function(err) {
                        log.error("Error while calling function [getOrganisationInUseHints]", err);
                    }));
        },

        showRemoveQuestion : function(organisationJoin) {
            var caption = string.substitute(i18n.organisationListAskRemoveCaption, {
                organisationName : organisationJoin.organisationName
            });
            var question = string.substitute(i18n.organisationListAskRemoveQuestion, {
                organisationName : organisationJoin.organisationName
            });

            InfoDialog.showQuestion({
                  title : caption,
                message : question,
                buttons : [
                           { type : InfoDialog.Button.YES, fct : lang.hitch(this, function() {
                               on.emit(this, "removeOrganisation", organisationJoin.organisationId);
                           })},                           
                           { type : InfoDialog.Button.NO }
                ]
            });            
        },

        showDeleteQuestion : function(organisationJoin) {
            var caption = string.substitute(i18n.organisationListAskDeleteCaption, {
                organisationName : organisationJoin.organisationName
            });
            var question = string.substitute(i18n.organisationListAskDeleteQuestion, {
                organisationName : organisationJoin.organisationName
            });

            InfoDialog.showQuestion({
                  title : caption,
                message : question,
                buttons : [
                           { type : InfoDialog.Button.YES, fct : lang.hitch(this, function() {
                               this.deleteOrganisation(organisationJoin);
                           })},                           
                           { type : InfoDialog.Button.NO }
                ]
            });
        },

        deleteOrganisation : function(organisationJoin) {
            var organisationPersonId = this.applicationContext.getPageContextOrganisationPersonId();

            var organisationService = this.applicationContext.getService("organisationService");
            this.registerAsyncOperationStarted(OrganisationListWidget.AsyncOperation.DELETE);
            organisationService.deleteOrganisation(organisationPersonId, organisationJoin.organisationId).then(
                lang.hitch(this, function() {
                    this.registerAsyncOperationFinished(OrganisationListWidget.AsyncOperation.DELETE);

                    var successMessage = string.substitute(i18n.organisationListWidgetDeleteOrganisationSuccess, {
                        organisationName : organisationJoin.organisationName
                    });
                    topic.publish("message/ok", successMessage);
                    on.emit(this, "discardSearchData");
                    on.emit(this, "doReloadSearchInfo");
					on.emit(this, "doSearch");
                }), lang.hitch(this, function(err) {
                    var failureMessage = string.substitute(i18n.organisationListWidgetDeleteOrganisationFailed, {
                        organisationName : organisationJoin.organisationName
                    });

                    ErrorHelper.processAsyncError({
                                   err : err,
                                widget : this,
                        asyncOperation : OrganisationListWidget.AsyncOperation.DELETE,
                                opName : "deleteOrganisation",
                               message : failureMessage
                    });
                })).otherwise(
                    lang.hitch(this, function(err) {
                        log.error("Error while calling function [deleteOrganisation]", err);
                    }));
        },

        showNotDeletableDialog : function(organisationJoin, hints) {
            var text = string.substitute(i18n.organisationListDeleteNotPossibleDialogText, {
                organisationName : organisationJoin.organisationName
            });

            var content = domConstruct.create("div", null, null);
            domClass.add(content, "dialogContent");
            DOMHelper.createTextNode("div", text, content);

            var ulNode = domConstruct.create("ul", null, content);
            for (var n = 0; n < hints.length; n++) {
                DialogHelper.addHint(ulNode, hints[n]);
            }

            var notDeletableDialog = new InfoDialog({
                        title : i18n.organisationListDeleteNotPossibleDialogCaption,
                      content : content,
                 defaultWidth : 750,
                defaultHeight : 350,
                      buttons : [ {    type : InfoDialog.Button.CLOSE } ]                                 
            });
            notDeletableDialog.show();            
        },

        constructTopDiv : function() {
            var grid = this.constructGrid({
                idProperty : "organisationId", 
                    gridId : this.gridId != null ? this.gridId : "OrganisationListGrid",
                 gridClass : "OnDemandGrid"
            });

            on(grid, "dgrid-sort", lang.hitch(this, function(e) {
                var organisationPersonId = this.applicationContext.getPageContextOrganisationPersonId();
                this.applicationContext.storeInLocalStorage([organisationPersonId], "/organisation/sort", e.sort);
            }));

            aspect.after(this.grid, "renderRow", lang.hitch(this, function(row, args) {
                var joinDto = args[0];
                this.updateRowClass(row, joinDto);
                return row;
            }));

            return grid;            
        },

        resize : function(newSize) {
            this.grid.resize(newSize);
            //this.updateColumns(this.columnSettings, touchedColumnId);
        },

        setData : function(params) {
            var organisationInfos = params.organisationInfos;

            this.organisationJoins = [];
            for (var n = 0; n < organisationInfos.length; n++) {
                var organisationInfo = organisationInfos[n];
                var organisationJoin = organisationInfo.organisationJoin;
                organisationJoin.actions = DataHelper.getAsSet(organisationInfo.actions);
                organisationJoin.networks = organisationInfo.networks;
                organisationJoin.networkIds = new Object();
                if (organisationJoin.networks != null) {
                    for (var z = 0; z < organisationJoin.networks.length; z++) {
                        organisationJoin.networkIds[organisationJoin.networks[z].id] = true;
                    }
                }                    

                this.organisationJoins.push(organisationJoin);
            }

            this.updateAuxiliaryData();

            this.reload();
        },

        setAdministratedNetworks : function(administratedNetworks) {
            this.administratedNetworks = administratedNetworks;
        },

        updateAuxiliaryData : function() {
            for (var n = 0; n < this.organisationJoins.length; n++) {
                var organisationJoin = this.organisationJoins[n];
                this.updateAuxiliaryDataForJoinDto(organisationJoin);
            }
        },

        updateAuxiliaryDataForJoinDto : function(organisationJoin) {
            organisationJoin.networks.sort(function(networkOne, networkTwo) {
                var nameOne = StringHelper.getNonNullString(networkOne.name);
                var nameTwo = StringHelper.getNonNullString(networkTwo.name);
                return nameOne.localeCompare(nameTwo);
            });

            organisationJoin._organisationString = organisationJoin.organisationName != null
                                                 ? organisationJoin.organisationName.trim() : null;
            organisationJoin._addressString = OrganisationHelper.getOrganisationAddressString(organisationJoin);
            organisationJoin._phoneString = this.getPhoneString(organisationJoin);
            organisationJoin._networksString = this.getNetworksString(organisationJoin);
			organisationJoin._notYetReferencedOrgs = [];
			
            organisationJoin._foreignNetwork = false;
            if (this.administratedNetworks != null && this.administratedNetworks.length > 0) {
                var organisationNetworkIds = organisationJoin.networkIds;
            	//check whether non administratedNetworks are referenced by organisation
                organisationJoin._nonReferencedToMyNetworks = !this.containsAll(this.administratedNetworks, organisationNetworkIds);
                var found = true;
                for (var n = 0; n < this.administratedNetworks.length; n++) {
                    var administratedNetworkId = this.administratedNetworks[n].id;
                    if (!(administratedNetworkId in organisationNetworkIds)){
                    	found = false;
                    	organisationJoin._notYetReferencedOrgs.push(this.administratedNetworks[n]);
                    }
                }
                organisationJoin._foreignNetwork = !found;
            }            
        },

        containsAll : function(array1, array2) {
        	//return this.administratedNetworks.every(v => !(v.id in organisationNetworkIds));
        	//return array1.every(i => array2.includes(i));
        	var containsAll = false;
	        for(var i=0; i < array1.length; i++)
	        {
	            if(array1[i].id in array2)
	                containsAll |= true;
                else
                	containsAll |= false;
	        }
		    return containsAll;        	
        },

        updateRowClass : function(domNode, organisationJoin) {
            domClass.remove(domNode, "personListForeignNetworkRow");

            if (organisationJoin._nonReferencedToMyNetworks) {
                domClass.add(domNode, "personListForeignNetworkRow");
            }
        },

        reload : function(params) {
        
            //this.updateColumns(this.columnSettings);
        
            this.setStoreData(this.organisationJoins);

            var organisationPersonId = this.applicationContext.getPageContextOrganisationPersonId();
            var oldSort = this.applicationContext.getFromLocalStorage([organisationPersonId], "/organisation/sort");

            if (oldSort == null) {
                // TODO                this.grid.set("sort", "_dueDateString", false);
                this.applicationContext.storeInLocalStorage([organisationPersonId], "/organisation/sort", this.grid.get("sort"));
            } else {
                this.grid.set("sort", oldSort);
            }           

            this.grid.updateSortArrow(this.grid.sort);
            this.grid.refresh();
        },

        getRowId : function(organisationJoin) {
            return organisationJoin.organisationId;
        },

        getQuickSearchTokensForRow : function(organisationJoin) {
            var tokens = [];
            tokens.push(organisationJoin._organisationString);
            tokens.push(organisationJoin._addressString);
            tokens.push(organisationJoin._phoneString);
            tokens.push(organisationJoin._netwoksString);
            return tokens;          
        },          

        destroy : function() {
            this.inherited(arguments);
        }
    });

    OrganisationListWidget.AsyncOperation = {
        ADD_TO_NETWORK : "AddToNetwork",
        GET_DELETE_INFO : "GetDeleteInfo",
        DELETE : "Delete"
    };

    return OrganisationListWidget;
});
