define(["clazzes/TinyLog",
	"clazzes/topic",
	"clazzes/util/EventHelper",
	"dojo/dom-class",
	"dojo/dom-construct",
	"dojo/Evented",
	"dojo/on",
	"dojo/_base/declare",
	"dojo/_base/lang"
],
	function(TinyLog,
		topic,
		EventHelper,
		domClass,
		domConstruct,
		Evented,
		on,
		declare,
		lang
	) {

		var className = "at.cdes.widget.util.DnDCellWidget";

		var log = new TinyLog(className);

		var DnDCellWidget = declare(className, Evented, {

			postCreate: function() {
				this.inherited(arguments);
				if (typeof this.rowObjectToString == "undefined") {
					throw new Error("Please set rowObjectToString");
				}
			},

			constructDnDButtonCell: function(parentDiv, iconClass) {
				// Icon for DnD the date into other table cells
				var dndButtonIcon = domConstruct.create("div", null, parentDiv);
				domClass.add(dndButtonIcon, "dndCellWidgetButtonCell");
				domClass.add(dndButtonIcon, "cdesWebIcon17x18 " + iconClass);

				on(dndButtonIcon, "mousedown", lang.hitch(this, function(e) {
					if (this.dndEnabled) {
						topic.publish(this.dndStartTopic, this.rowId, this.get("value"), e);
					}
					// Prevent the standard browser behaviour of displaying text etc. as selected, after DnD has started
					e.preventDefault();

				}));
				return dndButtonIcon;
			},

			setEditorValue: function(value, priorityChange) {
				// TODO
			},

			getEditorValue: function() {
				// TODO
			},

			emitValueChangeTopic: function(kindOfChange) {
				topic.publish(this.setDataTopic, this.domNode.id, this.get("value"), this, kindOfChange);
			},

			_setValueAttr: function(valueInfo, priorityChange) {
				// valueInfo: { value, id }

				if (log.isDebugEnabled()) {
					log.debug("setValue: ", valueInfo, priorityChange);
				}

				if (!this.domNode) {
					return;
				}

				if (valueInfo != null) {

					if (valueInfo.rowId != null) {
						this.rowId = valueInfo.rowId;

						if (log.isDebugEnabled()) {
							log.debug("=====> ... about to publish() ", this.setDataTopic);
						}

						if (priorityChange == null || priorityChange) {
							topic.publish(this.setDataTopic, this.domNode.id, valueInfo, this);
						}
					}
					if (valueInfo.options != null) {
						this.options = valueInfo.options;
						this.setEditorOptions(this.options, priorityChange);

						if (log.isDebugEnabled()) {
							log.debug("=====> ... about to publish() ", this.setDataTopic);
						}

						if (priorityChange == null || priorityChange) {
							topic.publish(this.setDataTopic, this.domNode.id, valueInfo, this);
						}
					}
					this.setEditorValue(valueInfo.value, priorityChange);

				} else {
					this.setEditorValue(null, priorityChange);
				}

				if (priorityChange == null || priorityChange) {
					// Emit a change event in order to trigger a dgrid-datachange event.  The latter one triggers
					// things like updating the documentNumberToDocuments map.
					on.emit(this, "change");
				}
			},

			_getValueAttr: function() {
				return {
					value: this.getEditorValue(),
					rowId: this.rowId,
					toString: this.rowObjectToString
				}
			},
		});

		DnDCellWidget.constructDnDInfo = function() {
			return {
				startY: null,
				startRowId: null,
				copiedValue: null,
				rowIdToOriginalValue: new Object(),
				domNodeIdToValueInfo: new Object(),
				rowIdToEditors: new Object(),
				rowIdToHolidayError: new Object(),
				rowIdToHolidayMessage: new Object(),
				mouseOverHandle: null,
				mouseUpHandle: null,
				mouseOutHandle: null
			};
		};

		DnDCellWidget.clearDnDInfo = function(dndInfo) {
			dndInfo.startY = null;
			dndInfo.startRowId = null;
			dndInfo.copiedValue = null;

			for (rowId in dndInfo.rowIdToOriginalValue) {
				delete dndInfo.rowIdToOriginalValue[rowId];
			}
			for (domNodeId in dndInfo.domNodeIdToValueInfo) {
				delete dndInfo.domNodeIdToValueInfo[domNodeId];
			}
			for (rowId in dndInfo.rowIdToEditors) {
				delete dndInfo.rowIdToEditors[rowId];
			}

			if (dndInfo.mouseOverHandle != null) {
				dndInfo.mouseOverHandle.remove();
				delete dndInfo.mouseOverHandle;
			}
			if (dndInfo.mouseUpHandle != null) {
				dndInfo.mouseUpHandle.remove();
				delete dndInfo.mouseUpHandle;
			}
			if (dndInfo.mouseOutHandle != null) {
				dndInfo.mouseOutHandle.remove();
				delete dndInfo.mouseOutHandle;
			}
		},

			DnDCellWidget.registerSetDataTopic = function(destroyableOwner, topicName, dndInfo, editorAttributeName) {
				var domNodeIdToValueInfo = dndInfo.domNodeIdToValueInfo;
				var rowIdToEditors = dndInfo.rowIdToEditors;

				destroyableOwner.own(
					topic.subscribe(topicName, lang.hitch(this, function(domNodeId, valueInfo, editor) {
						if (log.isDebugEnabled()) {
							log.debug("Received [" + topicName + "]: " + domNodeId + "; ", valueInfo);
						}

						domNodeIdToValueInfo[domNodeId] = valueInfo;

						if (!(valueInfo.rowId in rowIdToEditors)) {
							rowIdToEditors[valueInfo.rowId] = new Object();
						}
						rowIdToEditors[valueInfo.rowId][editorAttributeName] = editor;
					}), this)
				);
			};

		DnDCellWidget.getRowIdForEvent = function(domNodeIdToValueInfo, e) {
			var srcElement = EventHelper.getSrcElement(e);
			if (srcElement.id in domNodeIdToValueInfo) {
				return domNodeIdToValueInfo[srcElement.id].rowId;
			} else {
				for (var n = 0; n < srcElement.childNodes.length; n++) {
					if (srcElement.childNodes[n].id in domNodeIdToValueInfo) {
						return domNodeIdToValueInfo[srcElement.childNodes[n].id].rowId;
					}
				}
			}

			var currElement = srcElement;
			while (currElement.parentNode && currElement.parentNode.id != null) {
				currElement = currElement.parentNode;
				if (currElement.id && currElement.id in domNodeIdToValueInfo) {
					return domNodeIdToValueInfo[currElement.id].rowId;
				}
			}

			return null;
		};

		DnDCellWidget.registerMouseOverHandler = function(params) {

			var grid = params.grid;
			var columnId = params.columnId;
			var setter = params.setter;
			var offsetFilter = params.offsetFilter;
			var editorName = params.editorName;
			var domNodeIdToValueInfo = params.dndInfo.domNodeIdToValueInfo;
			var rowIdToEditors = params.dndInfo.rowIdToEditors;
			var startY = params.dndInfo.startY;
			var startRowId = params.dndInfo.startRowId;
			var copiedValue = params.dndInfo.copiedValue;
			var rowIdToOriginalValue = params.dndInfo.rowIdToOriginalValue;

			return on(grid, ".dgrid-cell.dgrid-column-" + columnId + ":mouseover", lang.hitch(this, function(e) {

				var searchedId = DnDCellWidget.getRowIdForEvent(domNodeIdToValueInfo, e);

				if (searchedId != null) {
					var srcElement = EventHelper.getSrcElement(e);
					if (log.isDebugEnabled()) {
						log.debug("mouseOver: " + srcElement.id, searchedId, domNodeIdToValueInfo);
					}

					var currY = e.clientY;
					var offset = (e.clientY < startY ? -1 : 1);
					var totalOffset = 0;

					var currRow = grid.row(startRowId);
					var prevRow = currRow;
					var rowIdsChangedThisTime = new Object();

					do {
						var currRowId = currRow.id;
						var editor = currRowId in rowIdToEditors ? rowIdToEditors[currRowId][editorName] : null;

						if (editor && currRowId != startRowId && (offsetFilter == null || offsetFilter(totalOffset))) {
							if (!(currRowId in rowIdToOriginalValue)) {
								rowIdToOriginalValue[currRowId] = editor.get("value").value;
							}

							rowIdsChangedThisTime[currRowId] = true;

							if (setter) {
								setter(currRowId, editor, copiedValue, totalOffset);
							} else {
								editor.set("value", { value: copiedValue });
							}
						}

						prevRow = currRow;
						currRow = grid.down(currRow.id, offset);
						totalOffset += offset;
					} while (prevRow != null && currRow != null && prevRow.id != currRow.id && prevRow.id != searchedId);

					for (var currRowId in rowIdToOriginalValue) {
						if (!(currRowId in rowIdsChangedThisTime)) {
							var originalValue = rowIdToOriginalValue[currRowId];
							var editor = currRowId in rowIdToEditors ? rowIdToEditors[currRowId][editorName] : null;
							editor.set("value", { value: originalValue });
						}
					}
				}
			}));
		};

		DnDCellWidget.startDnD = function(rowId, value, e, grid, dndInfo, columnId, setter, offsetFilter, editorName) {
			var srcElement = EventHelper.getSrcElement(e);

			value = value.value;

			if (dndInfo.mouseOverHandle != null) {
				dndInfo.mouseOverHandle.remove();
				delete dndInfo.mouseOverHandle;
			}

			dndInfo.startY = e.clientY;
			dndInfo.startRowId = rowId;
			dndInfo.copiedValue = value;
			dndInfo.rowIdToOriginalValue = new Object();

			dndInfo.mouseOutHandle = on(grid, "mouseout", lang.hitch(this, function(e) {
				if (srcElement == grid.domNode) {
					DnDCellWidget.stopDnD(dndInfo);
				}
			}));
			dndInfo.mouseUpHandle = on(document, "mouseup", lang.hitch(this, function(e) {
				DnDCellWidget.stopDnD(dndInfo);
			}));

			dndInfo.mouseOverHandle = DnDCellWidget.registerMouseOverHandler({
				grid: grid,
				columnId: columnId,
				setter: setter,
				offsetFilter: offsetFilter,
				editorName: editorName,
				dndInfo: dndInfo
			});
		};

		DnDCellWidget.stopDnD = function(dndInfo) {
			if (dndInfo.mouseOverHandle != null) {
				dndInfo.mouseOverHandle.remove();
				delete dndInfo.mouseOverHandle;
			}
			if (dndInfo.mouseUpHandle != null) {
				dndInfo.mouseUpHandle.remove();
				delete dndInfo.mouseUpHandle;
			}
			if (dndInfo.mouseOutHandle != null) {
				dndInfo.mouseOutHandle.remove();
				delete dndInfo.mouseOutHandle;
			}
		};

		DnDCellWidget.setupDnD = function(params) {
			var destroyableOwner = params.destroyableOwner;
			var setDataTopic = params.setDataTopic;
			var startDnDTopic = params.startDnDTopic;
			var editorName = params.editorName;
			var dndInfo = params.dndInfo;
			var grid = params.grid;
			var columnId = params.columnId;
			var setter = params.setter;
			var offsetFilter = params.offsetFilter;
			var onlySetData = params.onlySetData;

			DnDCellWidget.registerSetDataTopic(destroyableOwner, setDataTopic, dndInfo, editorName);

			if (!onlySetData) {
				destroyableOwner.own(
					topic.subscribe(startDnDTopic, lang.hitch(this, function(rowId, value, e) {
						DnDCellWidget.startDnD(rowId, value, e, grid, dndInfo, columnId, setter, offsetFilter, editorName);
					}), this)
				);
			}
		};

		return DnDCellWidget;
	});