define([
        "cdes/core/CdesVoc",
        "clazzes/TinyLog",
        "clazzes/util/StringHelper",
        "dojo/i18n!/cdes/nls/cdes-web-i18n.js"  
],
function(
         CdesVoc,
         TinyLog,
         StringHelper,
         i18n) {
    
    var className = "at.cdes.web.util.CodeHelper";
    
    var log = new TinyLog(className);
    
    var CodeHelper = {

        getObjectCodeFromCodes : function(objectCode, objectTypeCode) {
            return objectCode + (objectTypeCode ? objectTypeCode : "");
        },
        
        getObjectCode : function(joinDto) {
            return CodeHelper.getObjectCodeFromCodes(joinDto.objectCode, joinDto.objectTypeCode);
        },
        
        DocumentNumberComponent : {
            DOCUMENT_TYPE_CODE : "DocumentTypeCode",
            DOCUMENT_TYPE_GROUP_CODE : "DocumentTypeGroupCode",
            DOCUMENT_NUMBER : "DocumentNumber",
            DOCUMENT_PART : "DocumentPart",
            OBJECTPLANNER_CODE : "ObjectPlannerCode",
            OBJECTPLANNER_RELEASE_CODE : "ObjectPlannerReleaseCode",
            OBJECT_CODE : "ObjectCode",
            OBJECT_RELEASE_CODE : "ObjectReleaseCode",
            OBJECT_CODE_FULL : "ObjectCodeFull",                   // Object code and ObjectType code appended
            OBJECT_RELEASE_CODE_FULL : "ObjectReleaseCodeFull",
            OBJECT_TYPE_CODE : "ObjectTypeCode",
            OBJECT_RELEASE_TYPE_CODE : "ObjectReleaseTypeCode",
            SUBPROJECT_CODE : "SubprojectCode",
            SUBPROJECT_CODE_WITHOUTNUMBER : "SubprojectCodeWithoutNumber",
            SUBPROJECT_NUMBER : "SubprojectNumber",
            SUBPROJECT_TYPE_CODE : "SubprojectCodeType",
            PROJECT_CODE : "ProjectCode",
            PROJECT_PARTICIPATION_CUSTOMER_ID : "ProjectParticipationCustomerId"
        },
        
        getDocNumberComponentByOgnlPath : function(path) {
            switch(path) {
                case "document.documentType.code"                                : return CodeHelper.DocumentNumberComponent.DOCUMENT_TYPE_CODE; 
                case "document.documentType.documentTypeGroup.code"              : return CodeHelper.DocumentNumberComponent.DOCUMENT_TYPE_GROUP_CODE;              
                case "document.number"                                           : return CodeHelper.DocumentNumberComponent.DOCUMENT_NUMBER;               
				case "document.part"                                             : return CodeHelper.DocumentNumberComponent.DOCUMENT_PART;               
                case "document.objectPlanner.code"                               : return CodeHelper.DocumentNumberComponent.OBJECTPLANNER_CODE;                    
                case "document.objectPlanner.object.code"                        : return CodeHelper.DocumentNumberComponent.OBJECT_CODE_FULL;
                case "document.objectPlanner.object.codeWithoutType"             : return CodeHelper.DocumentNumberComponent.OBJECT_CODE;
                case "document.objectPlanner.object.objectType.code"             : return CodeHelper.DocumentNumberComponent.OBJECT_TYPE_CODE;              
                case "document.objectPlanner.planner.customerId"                 : return CodeHelper.DocumentNumberComponent.PROJECT_PARTICIPATION_CUSTOMER_ID;
                case "document.documentList.subProject.code"                     : return CodeHelper.DocumentNumberComponent.SUBPROJECT_CODE;
                case "document.documentList.subProject.codeWithoutNumber"        : return CodeHelper.DocumentNumberComponent.SUBPROJECT_CODE_WITHOUTNUMBER;
                case "document.documentList.subProject.number"                   : return CodeHelper.DocumentNumberComponent.SUBPROJECT_NUMBER;
                case "document.documentList.subProject.subProjectType.code"      : return CodeHelper.DocumentNumberComponent.SUBPROJECT_TYPE_CODE;
                case "document.documentList.subProject.project.code"             : return CodeHelper.DocumentNumberComponent.PROJECT_CODE;
                case "relObjectPlanner.code"                                     : return CodeHelper.DocumentNumberComponent.OBJECTPLANNER_RELEASE_CODE;                    
                case "relObjectPlanner.object.code"                              : return CodeHelper.DocumentNumberComponent.OBJECT_RELEASE_CODE_FULL;
                case "relObjectPlanner.object.codeWithoutType"                   : return CodeHelper.DocumentNumberComponent.OBJECT_RELEASE_CODE;
                case "relObjectPlanner.object.objectType.code"                   : return CodeHelper.DocumentNumberComponent.OBJECT_RELEASE_TYPE_CODE;
                case "relObjectPlanner.planner.customerId"                       : return CodeHelper.DocumentNumberComponent.PROJECT_PARTICIPATION_CUSTOMER_ID;
                case "relObjectPlanner.objectPlanner.planner.customerId"         : return CodeHelper.DocumentNumberComponent.PROJECT_PARTICIPATION_CUSTOMER_ID;
                default : throw new Error("Trying to resolve unsupported ognl path: ", path);
            }
        },
        
		//the documentNumberUniqueness is controlled by settings in the DocumentNumberPart's ognlRule
		//there should be at least an ognlRule for 'document.number' with uniqueKey=true
		//3 cases:
		// - uniqueness within objectPlanner: ognlRule contains 'objectPlanner' and its uniqueKey=true
		// - uniqueness within object: ognlRule contains 'object' and its uniqueKey=true
		// - uniqueness within subPproject: any other case
        getDocumentNumberUniqueness : function(documentNumberParts) {
            for (var n = 0; n < documentNumberParts.length; n++) {
                var part = documentNumberParts[n];

                // Condition based on at.cdes.controller.documentNumber.DocumentNumberRegistry.isObjectPlannerUniqueKey(Project) in old code.
                if (part.ognlRule && part.ognlRule.indexOf(".objectPlanner.") != -1 && part.uniqueKey) {
                    return CdesVoc.DocumentNumberUniqueness.OBJECT_PLANNER;
                } else if (part.ognlRule && part.ognlRule.indexOf(".object.") != -1 && part.uniqueKey) {
                    return CdesVoc.DocumentNumberUniqueness.OBJECT;
                }
            }
            return null;
        },
        
        formatStringCode : function(format, code) {
            if (code == null) {
                return "";
            } else if (format == null || format.length == 0) {
                return code;
            } else {
                // Very poor mans version of String.format.
                
                // format strings of the form %.4s
                if (format.length == 4 && format.charAt(0) == '%' && format.charAt(1) == '.' && format.charAt(3) == 's') {
                    var length = parseInt(format.charAt(2));
                    
                    var result = code;
                    if (code.length > 4) {
                        return code.substr(0, 4);
                    } else {
                        return code;
                    }
                } else {
                    return code;
                }
            }               
        },
        
        formatIntegerCode : function(format, code) {
            if (code == null) {
                return "";
            } else if (format == null || format.length == 0) {
                return code.toString();
            } else {
                // format strings of the form %04d
                if (format.length == 4 && format.charAt(0) == '%' && format.charAt(1) == '0' && format.charAt(3) == 'd') {
                    var length = parseInt(format.charAt(2));
                    
                    var result = code.toString();
                    while (result.length < length) {
                        result = "0" + result;
                    }
                    return result;
                // format strings of the form %.1d eg 21 -> 2.1
                } else if (format.length == 4 && format.charAt(0) == '%' && format.charAt(1) == '.' && format.charAt(3) == 'd') {
                    var prec = parseInt(format.charAt(2));
                    var codeStr = code.toString();
                    var part1 = codeStr.substr(0, codeStr.length-prec);
                    var part2 = codeStr.substr(codeStr.length-prec, prec);
                    return part1 + '.' + part2;
                // format strings of the form pre%xxdpost eg %d.0: 2 -> 2.0
                } else if (format.indexOf('%') >= 0 && format.indexOf('d') > format.indexOf('%') && (format.indexOf('d')-format.indexOf('%')) < 4) {
                	var length = format.length;
               		result = code.toString();
                	var pre = '';
                	var post = '';
                	var start = format.indexOf('%');
                	if (start > 0)
                		pre = format.substr(0, start);
                	
                	var end = format.indexOf('d');
                	if (end > 0 && format.length > end)
                		post = format.substr(end+1, format.length-1);
                	
                	if (end == start+2){ // %4d
                		result = code.toString();
                		var length = parseInt(format.charAt(start+1));
                    	while (result.length < length) 
                        	result = " " + result;
                	}
                	if (end == start+3){ // %04d
                		result = code.toString();
                		var length = parseInt(format.charAt(start+2));
                    	while (result.length < length) 
                        	result = "0" + result;
                	}
                	return pre + result + post;
                
                } else {
                    return code;
                }
            }               
        },
        
        formatFixedLengthIntegerCode : function(code, length) {
            var result = code != null?code.toString():'0';
            while (result.length < length) {
                result = "0" + result;
            }
            return result;
        },
        
        getDocumentNumberParts : function(applicationContext) {
            var project = applicationContext.getPageContextProject();
            var documentNumberPartGroupId = project ? project.documentNumberPartGroupId : null;
            
            if (documentNumberPartGroupId != null) {
                var documentNumberGroup = applicationContext.getDocumentNumberGroup(documentNumberPartGroupId);
                
                if (documentNumberGroup != null) {
                    var masterData = applicationContext.getMasterData(documentNumberGroup.masterDataSetId);
                    var numberParts = (masterData && documentNumberPartGroupId in masterData.groupIdToDocumentNumberParts 
                        ? masterData.groupIdToDocumentNumberParts[documentNumberPartGroupId] : null);
                    
                    return numberParts;
                } else {
                    return [];
                }
            } else {
                return [];
            }
        },

		getDocumentPartNameByNumberPart : function(documentNumberInfo, numberPart) {
			
			var token = "";
			var component = CodeHelper.getDocNumberComponentByOgnlPath(numberPart.ognlRule);
			
			switch(component) {
				case CodeHelper.DocumentNumberComponent.DOCUMENT_TYPE_CODE:
		            token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.documentType?documentNumberInfo.documentType.code:"");                                   
					break;
		        case CodeHelper.DocumentNumberComponent.DOCUMENT_TYPE_GROUP_CODE:
		            token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.documentTypeGroup ? documentNumberInfo.documentTypeGroup.code : "");
					break;
		        case CodeHelper.DocumentNumberComponent.DOCUMENT_NUMBER:
		            token = CodeHelper.formatFixedLengthIntegerCode(documentNumberInfo.number, numberPart.length != null ? numberPart.length : 4);
					break;
		        case CodeHelper.DocumentNumberComponent.DOCUMENT_PART:
					token = CodeHelper.formatFixedLengthIntegerCode(documentNumberInfo.part, numberPart.length != null ? numberPart.length : 4);
					break;
				case CodeHelper.DocumentNumberComponent.OBJECTPLANNER_CODE:
		            token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectPlannerCode);
					break;
		        case CodeHelper.DocumentNumberComponent.OBJECT_CODE:
		            token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectCode);
					break;
		        case CodeHelper.DocumentNumberComponent.OBJECT_CODE_FULL:
		            token = (documentNumberInfo.objectType != null
		                ? CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectCode + documentNumberInfo.objectType.code)
		                : CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectCode));
					break;
		        case CodeHelper.DocumentNumberComponent.OBJECT_TYPE_CODE:
		            token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectType.code);
					break;
		        case CodeHelper.DocumentNumberComponent.PROJECT_PARTICIPATION_CUSTOMER_ID:
		            token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.projectParticipationCustomerId);
					break;
		        case CodeHelper.DocumentNumberComponent.SUBPROJECT_CODE:
		            token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.subProjectCode);
					break;
		        case CodeHelper.DocumentNumberComponent.SUBPROJECT_CODE_WITHOUTNUMBER:
		            token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.subProjectCode);
					break;
		        case CodeHelper.DocumentNumberComponent.SUBPROJECT_NUMBER:
					token = documentNumberInfo.subProjectNumber != null ? CodeHelper.formatIntegerCode(numberPart.format, documentNumberInfo.subProjectNumber) : "";
					break;
		        case CodeHelper.DocumentNumberComponent.SUBPROJECT_TYPE_CODE:
		            token = (documentNumberInfo.subProjectType != null ? CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.subProjectType.code) : "");                                      
					break;
		        case CodeHelper.DocumentNumberComponent.PROJECT_CODE:
		            token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.projectCode);
					break;
		        case CodeHelper.DocumentNumberComponent.OBJECTPLANNER_RELEASE_CODE:
		            token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectPlannerReleaseCode)
					break;
		        case CodeHelper.DocumentNumberComponent.OBJECT_RELEASE_CODE_FULL:
		            token = (documentNumberInfo.objectType != null
		                ? CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectReleaseCode + documentNumberInfo.objectType.code)
		                : CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectReleaseCode));
					break;
		        case CodeHelper.DocumentNumberComponent.OBJECT_RELEASE_CODE:
		            token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectReleaseCode);
					break;
		        case CodeHelper.DocumentNumberComponent.OBJECT_RELEASE_TYPE_CODE:
		            token = (documentNumberInfo.objectReleaseObjectType  != null ? CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectReleaseObjectType.code) : "");
		        default:
					log.warn("Found unknown component of numberpart: ", component);
	        }                       
		    return token;
		},
        
        getDocumentName : function(documentNumberInfo) {
			
            var documentName = "";
			
            var numberParts = documentNumberInfo.numberParts;
            for (var n = 0; n < numberParts.length; n++) {
                var numberPart = numberParts[n];
                var component = CodeHelper.getDocNumberComponentByOgnlPath(numberPart.ognlRule);
                var token = "";
                if (component == CodeHelper.DocumentNumberComponent.DOCUMENT_TYPE_CODE) {
                    token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.documentType.code);                                   
                } else if (component == CodeHelper.DocumentNumberComponent.DOCUMENT_TYPE_GROUP_CODE) {
                    token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.documentTypeGroup ? documentNumberInfo.documentTypeGroup.code : "");
                } else if (component == CodeHelper.DocumentNumberComponent.DOCUMENT_NUMBER) {
                    token = CodeHelper.formatFixedLengthIntegerCode(documentNumberInfo.number, numberPart.length != null ? numberPart.length : 4);
                } else if (component == CodeHelper.DocumentNumberComponent.DOCUMENT_PART) {
					token = CodeHelper.formatFixedLengthIntegerCode(documentNumberInfo.part, numberPart.length != null ? numberPart.length : 4);
				} else if (component == CodeHelper.DocumentNumberComponent.OBJECTPLANNER_CODE) {
                    token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectPlannerCode);
                } else if (component == CodeHelper.DocumentNumberComponent.OBJECT_CODE) {
                    token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectCode);
                } else if (component == CodeHelper.DocumentNumberComponent.OBJECT_CODE_FULL) {
                    token = (documentNumberInfo.objectType != null
                        ? CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectCode + documentNumberInfo.objectType.code)
                        : CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectCode));
                } else if (component == CodeHelper.DocumentNumberComponent.OBJECT_TYPE_CODE) {
                    token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectType.code);
                } else if (component == CodeHelper.DocumentNumberComponent.PROJECT_PARTICIPATION_CUSTOMER_ID) {
                    token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.projectParticipationCustomerId);
                } else if (component == CodeHelper.DocumentNumberComponent.SUBPROJECT_CODE) {
                    token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.subProjectCode);
                } else if (component == CodeHelper.DocumentNumberComponent.SUBPROJECT_CODE_WITHOUTNUMBER) {
                    token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.subProjectCode);
                } else if (component == CodeHelper.DocumentNumberComponent.SUBPROJECT_NUMBER) {
					token = documentNumberInfo.subProjectNumber != null ? CodeHelper.formatIntegerCode(numberPart.format, documentNumberInfo.subProjectNumber) : "";
                } else if (component == CodeHelper.DocumentNumberComponent.SUBPROJECT_TYPE_CODE) {
                    token = (documentNumberInfo.subProjectType != null ? CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.subProjectType.code) : "");                                      
                } else if (component == CodeHelper.DocumentNumberComponent.PROJECT_CODE) {
                    token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.projectCode);
                } else if (component == CodeHelper.DocumentNumberComponent.OBJECTPLANNER_RELEASE_CODE) {
                    token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectPlannerReleaseCode)
                } else if (component == CodeHelper.DocumentNumberComponent.OBJECT_RELEASE_CODE_FULL) {
                    token = (documentNumberInfo.objectType != null
                        ? CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectReleaseCode + documentNumberInfo.objectType.code)
                        : CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectReleaseCode));
                } else if (component == CodeHelper.DocumentNumberComponent.OBJECT_RELEASE_CODE) {
                    token = CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectReleaseCode);
                } else if (component == CodeHelper.DocumentNumberComponent.OBJECT_RELEASE_TYPE_CODE) {
                    token = (documentNumberInfo.objectReleaseObjectType  != null ? CodeHelper.formatStringCode(numberPart.format, documentNumberInfo.objectReleaseObjectType.code) : "");
                } else {
                    throw new Error("Unknown numberPart component: ", numberPart);
                }                       
                
                documentName += token;
                
                if (numberPart.seperator != null && n < numberParts.length - 1) {
                    documentName += numberPart.seperator;
                }
            }
            
            return documentName;
        },
        
        getUniqueKeyForDocument : function(document, numberParts, idToDocumentType, idToObjectType, projectParticipation, object) {
            var key = "";
			var noUniqueKeyAtAll = true;
            for (var n = 0; n < numberParts.length; n++) {
                var numberPart = numberParts[n];
                if (numberPart.uniqueKey) {
					noUniqueKeyAtAll = false;
                    var ognlRule = numberPart.ognlRule;
                    if (ognlRule == "document.documentType.documentTypeGroup.code") {
                        if (document.documentTypeId != null) {
                            var documentType = idToDocumentType[document.documentTypeId];
                            
                            if (documentType != null && documentType.documentTypeGroupId != null) {
                                var documentTypeGroup = idToDocumentType[documentType.documentTypeGroupId];
                                
                                if (documentTypeGroup != null && documentTypeGroup.code != null) {
                                    key = CodeHelper.appendToUniqueKey(key, documentTypeGroup.code);
                                } else {
                                    key = CodeHelper.appendToUniqueKey(key, "---");
                                }
                            } else {
                                key = CodeHelper.appendToUniqueKey(key, "---");
                            }
                        } else {
                            key = CodeHelper.appendToUniqueKey(key, "---");
                        }
                    } else if (ognlRule == "document.documentType.code") {
                        if (document.documentTypeId != null) {
                            var documentType = idToDocumentType[document.documentTypeId];
                            if (documentType != null && documentType.code != null) {
                                key = CodeHelper.appendToUniqueKey(key, documentType.code);
                            } else {
                                key = CodeHelper.appendToUniqueKey(key, "---");
                            }
                        } else {
                            key = CodeHelper.appendToUniqueKey(key, "---");                                         
                        }
                    } else if (ognlRule == "relObjectPlanner.planner.customerId") {
                        if (projectParticipation != null && projectParticipation.customerId != null) {
                            key = CodeHelper.appendToUniqueKey(key, projectParticipation.customerId);
                        } else {
                            key = CodeHelper.appendToUniqueKey(key, "---");
                        }
                	} else if (ognlRule == "relObjectPlanner.object.codeWithoutType") {
                  		if (object != null && object.code != null) {
                          	key = CodeHelper.appendToUniqueKey(key, object.code);
                      	} else {
                          	key = CodeHelper.appendToUniqueKey(key, "---");
						}
					} else if (ognlRule == "relObjectPlanner.object.objectType.code") {
                       if (object != null && object.code != null) {
							objectType = idToObjectType[object.objectTypeId];
               				key = CodeHelper.appendToUniqueKey(key, objectType.code);
                       } else {
                    		key = CodeHelper.appendToUniqueKey(key, "---");
                       }
                   } else if (ognlRule == "document.number") {
                        key = CodeHelper.appendToUniqueKey(key, document.number.toString());
                    } else if (ognlRule == "document.part" && document.part != null) {
                        key = CodeHelper.appendToUniqueKey(key, document.part.toString());
                    }
                }
            }
            
            return noUniqueKeyAtAll?null:key;
        },
        
        appendToUniqueKey : function(uniqueKey, suffix) {
            if (uniqueKey.length == 0) {
                return uniqueKey + suffix;
            } else {
                return uniqueKey + "___" + suffix;
            }
        },

        getI18nDueDateRemark : function(dueDateRemark) {
            if(dueDateRemark == null) {
                return null;
            } else if (dueDateRemark == "noEntries") {
                return i18n.dueDateRemarkNoEntries;
            } else if (dueDateRemark == "beforeConstruction") {
                return i18n.dueDateRemarkBeforeConstruction;
            } else if (dueDateRemark == "whileConstruction") {
                return i18n.dueDateRemarkWhileConstruction;
            } else if (dueDateRemark == "duringConstruction") {
                return i18n.dueDateRemarkDuringConstruction;
            } else if (dueDateRemark == "beforeStarting") {
                return i18n.dueDateRemarkBeforeStarting;
            } else if (dueDateRemark == "afterConstruction") {
                return i18n.dueDateRemarkAfterConstruction;
            } else if (dueDateRemark == "inOperation") {
                return i18n.dueDateRemarkInOperation;
            } else if (dueDateRemark == "tillOperationApproval") {
                return i18n.dueDateRemarkTillOperationApproval;
            } else {
                // Note: We migrated raw strings instead of the symbolic ones, thus this case will be needed, at least until we fix this.
                return dueDateRemark;
            }           
        },

        getDueDateRemarkOptions : function(defaultEntry) {
            if (defaultEntry == null) {
                defaultEntry = { value : "noEntries", label : i18n.dueDateRemarkNoEntries };
            }                

            var options = [defaultEntry,
                { value : "beforeConstruction", label : i18n.dueDateRemarkBeforeConstruction },
                { value : "whileConstruction", label : i18n.dueDateRemarkWhileConstruction },
                { value : "duringConstruction", label : i18n.dueDateRemarkDuringConstruction },
                { value : "beforeStarting", label : i18n.dueDateRemarkBeforeStarting },
                { value : "afterConstruction", label : i18n.dueDateRemarkAfterConstruction },
                { value : "inOperation", label : i18n.dueDateRemarkInOperation },
                { value : "tillOperationApproval", label : i18n.dueDateRemarkTillOperationApproval }
            ];
            return options;         
        },

        getDocumentVersionStatusOptions : function() {
            var options = [
                { label : i18n.whatever, value : CdesVoc.WHATEVER },
                { value : CdesVoc.DocumentVersionStatus.INREVIEW, label : i18n.documentVersionStateInReview },
                { value : CdesVoc.DocumentVersionStatus.RELEASEDPOSITIV, label : i18n.documentVersionStateReleasedPositiv },
	        { value : CdesVoc.DocumentVersionStatus.RELEASEDNEGATIV, label : i18n.documentVersionStateReleasedNegativ },
                { value : CdesVoc.DocumentVersionStatus.INVALIDATEDVERSION, label : i18n.documentVersionStateInvalidatedVersion },
	        { value : CdesVoc.DocumentVersionStatus.INVALIDATEDALL, label : i18n.documentVersionStateInvalidatedAll },
	        { value : CdesVoc.DocumentVersionStatus.NOTINREVIEW, label : i18n.documentVersionStateNotInReview },
	        { value : CdesVoc.DocumentVersionStatus.DELETED, label : i18n.documentVersionStateDeleted },
	        { value : CdesVoc.DocumentVersionStatus.PREREVIEWED, label : i18n.documentVersionStatePrereviewed },
                { value : CdesVoc.DocumentVersionStatus.NOTUPLOADED, label : i18n.documentVersionStatusNotUploaded },
	        { value : CdesVoc.DocumentVersionStatus.ACTIVE, label : i18n.documentVersionStateActive },
	        { value : CdesVoc.DocumentVersionStatus.ASBUILT, label : i18n.documentVersionStateAsBuilt }
            ];
            return options;    
        },

        getDocumentVersionStatusString : function(documentVersionStatus) {
            if (documentVersionStatus == CdesVoc.DocumentVersionStatus.INREVIEW) {
                return i18n.documentVersionStateInReview;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.RELEASEDPOSITIV) {
                return i18n.documentVersionStateReleasedPositiv;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.NOTINREVIEW) {
                return i18n.documentVersionStateNotInReview;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.RELEASEDNEGATIV) {
                return i18n.documentVersionStateReleasedNegativ;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.INVALIDATEDVERSION) {
                return i18n.documentVersionStateInvalidatedVersion;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.INVALIDATEDALL) {
                return i18n.documentVersionStateInvalidatedAll;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.DELETED) {
                return i18n.documentVersionStateDeleted;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.PREREVIEWED) {
                return i18n.documentVersionStatePrereviewed;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.ACTIVE) {
                return i18n.documentVersionStateActive;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.ASBUILT) {
                return i18n.documentVersionStateAsBuilt;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.NOTUPLOADED) {
                return i18n.documentVersionStatusNotUploaded;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.REPLACED) {
                return i18n.documentVersionStatusReplaced;
            } else if (documentVersionStatus == CdesVoc.DocumentVersionStatus.ABORTED) {
                return i18n.documentVersionStatusAborted;
            } else {
                return i18n.documentVersionStatusNoStatus;
            }                
        },

        isDocumentVersionStatusInAction : function(documentVersionStatus) {
            return documentVersionStatus == CdesVoc.DocumentVersionStatus.INREVIEW
            || documentVersionStatus == CdesVoc.DocumentVersionStatus.INVALIDATEDVERSION
            || documentVersionStatus == CdesVoc.DocumentVersionStatus.RELEASEDNEGATIV
            || documentVersionStatus == CdesVoc.DocumentVersionStatus.DELETED;
        },            

        getDocumentVersionPart : function(joinDto, prefix) {
            return joinDto[prefix + "VersionParta"]
                + joinDto[prefix + "VersionPartSeperator"]
                + joinDto[prefix + "VersionPartb"];
        },

        getReviewCycleStatusSearchDelayOptions : function() {
            var options = [
                { label : i18n.whatever, value : CdesVoc.WHATEVER },
                { value : CdesVoc.ReviewCycleStatusSearchDelay.NO_DELAY, label : i18n.reviewCycleStatusSearchNoDelay },
                { value : CdesVoc.ReviewCycleStatusSearchDelay.DELAY, label : i18n.reviewCycleStatusSearchDelay },
                { value : CdesVoc.ReviewCycleStatusSearchDelay.LONG_DELAY, label : i18n.reviewCycleStatusSearchLongDelay }
            ];
            return options;            
        },

        getBooleanYesNo : function(b) {
            return (b ? i18n.yes : i18n.no);   
        },

        getMailSendModeByCode : function(code) {
            if (code == 1) {
                return CdesVoc.MailSendMode.NEVER.value;
            } else if (code == 2) {
                return CdesVoc.MailSendMode.BRIEF.value;
            } else if (code == 3) {
                return CdesVoc.MailSendMode.ALWAYS.value;
            } else {
                throw new Error("Unsupported MailSendMode code [" + code + "] possibly loaded from database encountered.  Please inspect.");
            }                
        },

        getCodeByEmailSendMode : function(emailSendMode) {
            if (emailSendMode == CdesVoc.MailSendMode.NEVER.value) {
                return 1;
            } else if (emailSendMode == CdesVoc.MailSendMode.BRIEF.value) {
                return 2;
            } else if (emailSendMode == CdesVoc.MailSendMode.ALWAYS.value) {
                return 3;
            } else {
                return null;
            }
        },

    };
    
    return CodeHelper;
});
