/***********************************************************
* $Id$
*
* Copyright (C) 2017 ev-i Informationstechnologie Gmbh
*
**********************************************************/

define([ 
    "cdes/document/DocumentCompareCanvasWidget",
    "cdes/document/DocumentCompareOverviewWidget",
    "clazzes/TinyLog",
    "clazzes/canvas/CanvasHelper",
    "clazzes/canvas/image/TiledImage",
    "clazzes/util/DOMHelper",
    "clazzes/util/ErrorHelper",
    "clazzes/util/WidgetHelper",
    "clazzes/widgets/layout/ContentWidget",
    "dijit/form/Select",
    "dojo/dom-class",
    "dojo/dom-construct",
    "dojo/dom-style",
    "dojo/on",
    "dojo/string",
    "dojo/_base/declare",
    "dojo/_base/lang",
    "dojo/i18n!/cdes/nls/cdes-web-i18n.js"
], function(
    DocumentCompareCanvasWidget,
    DocumentCompareOverviewWidget,
    TinyLog,
    CanvasHelper,
    TiledImage,
    DOMHelper,
    ErrorHelper,
    WidgetHelper,
    ContentWidget,
    Select,    
    domClass,
    domConstruct,
    domStyle,
    on,
    string,    
    declare,
    lang,
    i18n
) {

//// css-prefix = documentVersionCompare
//// i18n-prefix = documentVersionCompare

    var className = "at.cdes.web.document.DocumentComparePage";

    var log = new TinyLog(className);

    var DocumentComparePage = declare(className, ContentWidget, {

        constructor : function(params) {
            lang.mixin(this, params);

            this.topDiv = this.constructTopDiv();

            this.registerAsyncOperationStarted(DocumentComparePage.AsyncOperation.LOAD_VERSIONS);

            var documentService = this.applicationContext.getService("documentService");

			var organisationPersonId = this.applicationContext.getPageContextOrganisationPersonId();
            documentService.getDocumentCompareData(organisationPersonId, this.documentVersionId).then(
                lang.hitch(this, function(documentCompareData) {
                    this.registerAsyncOperationFinished(DocumentComparePage.AsyncOperation.LOAD_VERSIONS);

                    this.initialize(documentCompareData);
                }),
                lang.hitch(this, function(err) {
                    ErrorHelper.processAsyncError({
                        err : err,
                        widget : this,
                        asyncOperation : DocumentComparePage.AsyncOperation.LOAD_VERSIONS,
                                opName : "getDocumentCompareData",
                               message : i18n.documentVersionCompareGetDocumentCompareDataFailed
                    });
                })).otherwise(
                    lang.hitch(this, function(err) {
                        log.error("Error while calling function [getDocumentCompareData]", err);
                    }));


            this.allFieldsValid = true;
        },

        getWidgetId : function() {
            return "DocumentComparePage";
        },

        getDataId : function() {
            return null;
        },

        getContainer : function() {
            return this.topDiv;
        },

        constructTopDiv : function() {
            var topDiv = domConstruct.create("div", null, null);

            // Caption Bar
            this.captionBarDiv = this.constructCaptionBar();
            domClass.add(this.captionBarDiv, "documentComparePageCaptionBar");
            domConstruct.place(this.captionBarDiv, topDiv);

            // Content
            this.contentDiv = this.constructContentDiv();
            domConstruct.place(this.contentDiv, topDiv);

            return topDiv;
        },

        constructCaptionBar : function() {
            var captionBarDiv = domConstruct.create("div", null, null);

            // Caption
            var caption = string.substitute(i18n.documentComparePageCaption, {
                baseDocumentName : ""
            });                
            
            this.captionDiv = DOMHelper.createTextNode("span", caption, captionBarDiv, "documentComparePageCaption");

            this.documentVersionSelect = new Select({
                label : i18n.documentVersionCompareVersionLabel,
                title : i18n.documentVersionCompareVersionToolTip                
            });
            domClass.add(this.documentVersionSelect.domNode, "documentCompareVersionSelect");
            domConstruct.place(this.documentVersionSelect.domNode, captionBarDiv);
            WidgetHelper.handleSelectEvents(this.documentVersionSelect, lang.hitch(this, this.updateSubWidgets));

            return captionBarDiv;
        },

        constructContentDiv : function() {
            var contentDiv = domConstruct.create("div", null, null);
            domClass.add(contentDiv, "refNodeOfPositionAbsolute documentComparePageContentDiv");

            this.canvasWidget = new DocumentCompareCanvasWidget({
                applicationContext : this.applicationContext
            });                
            var canvasNode = this.canvasWidget.getContainer();
            domClass.add(canvasNode, "documentCompareCanvasNode");
            domConstruct.place(canvasNode, contentDiv);

            this.overviewWidget = new DocumentCompareOverviewWidget({
                applicationContext : this.applicationContext
            });
            var overviewCanvasNode = this.overviewWidget.getContainer();
            domClass.add(overviewCanvasNode, "fixedDialogWidget documentCompareOverviewCanvasNode");            
            domConstruct.place(overviewCanvasNode, contentDiv);            
            on(this.overviewWidget, "needResize", lang.hitch(this, this.resizeOverviewWidget));

            return contentDiv;
        },

        initialize : function(documentCompareData) {
            this.ownDocumentVersion = documentCompareData.ownDocumentVersion;
            this.viewableVersions = documentCompareData.viewableVersions;
            this.viewableVersions.sort(function(versionOne, versionTwo) {
                return versionOne.uploaded - versionTwo.uploaded;
            });                        

            DOMHelper.setInnerText(this.captionDiv, string.substitute(i18n.documentComparePageCaption, {
                baseDocumentName : this.ownDocumentVersion.name
            }));

            var viewableVersionOptions = [];
            this.idToViewableVersion = new Object();
            for (var n = 0; n < this.viewableVersions.length; n++) {
                viewableVersionOptions.push({ value : this.viewableVersions[n].id, label : this.viewableVersions[n].name });
                this.idToViewableVersion[this.viewableVersions[n].id] = this.viewableVersions[n];
            }
            WidgetHelper.setSelectOptionsAndKeepLabelIfPossible(this.documentVersionSelect, viewableVersionOptions);
            if (viewableVersionOptions.length > 0) {
                this.documentVersionSelect.set("value", viewableVersionOptions[0].value);
            }            

            this.updateSubWidgets();
            this.resizeOverviewWidget();
        },

        updateSubWidgets : function() {
            var chosenDocumentVersion = this.getChosenDocumentVersion();
            var chosenDocumentVersionId = chosenDocumentVersion != null ? chosenDocumentVersion.id : null;
            if (this.lastChosenDocumentVersionId == null || this.lastChosenDocumentVersionId != chosenDocumentVersion.id) {
                this.lastChosenDocumentVersionId = chosenDocumentVersionId;

                if (this.tiledImage == null) {
                    // Set up TiledImage.  Compare in old code:
                    // at.cdes.tapestry.components.links.CmpDVTiledPNGImage.getImageSetup(IRequestCycle)
                    // at.cdes.tapestry.components.links.TiledPNGImage.renderComponent(IMarkupWriter, IRequestCycle)            
                    this.tiledImage = new TiledImage({
                                 xSize : this.getXSize(),
                                 ySize : this.getYSize(),
                              baseSize : DocumentComparePage.DEFAULT_BASE_SIZE,
                            resolution : 72.0,   
                                 angle : this.ownDocumentVersion.previewRotation,
                             thumbSize : DocumentComparePage.DEFAULT_THUMB_SIZE,
                          canvasWidget : this.canvasWidget,
                        overviewWidget : this.overviewWidget                
                    });
                } else {
                    this.tiledImage.clear({
                             xSize : this.getXSize(),
                             ySize : this.getYSize(),
                          baseSize : DocumentComparePage.DEFAULT_BASE_SIZE,
                        resolution : 72.0,   
                             angle : this.ownDocumentVersion.previewRotation                    
                    });
                }                    

                this.canvasWidget.setData({
                       ownDocumentVersion : this.ownDocumentVersion,
                    chosenDocumentVersion : this.getChosenDocumentVersion(),                
                               tiledImage : this.tiledImage
                });
                this.overviewWidget.setData({
                       ownDocumentVersion : this.ownDocumentVersion,
                    chosenDocumentVersion : this.getChosenDocumentVersion(),                
                               tiledImage : this.tiledImage
                });
                this.tiledImage.move(0, 0);                
            }
        },

        getChosenDocumentVersion : function() {
            var chosenDocumentVersionId = WidgetHelper.getSelectValue(this.documentVersionSelect);
            return chosenDocumentVersionId != null ? this.idToViewableVersion[chosenDocumentVersionId] : null;            
        },

        getRotation : function(documentVersion) {
            return documentVersion.previewRot != null ? documentVersion.previewRot : 0;
        },

        getDeltaAngle : function(chosenDocumentVersion) {
            var chosenDocumentVersionRotation = this.getRotation(chosenDocumentVersion);
            var ownDocumentVersionRotation = this.getRotation(this.ownDocumentVersion);
            return Math.abs(chosenDocumentVersionRotation - ownDocumentVersionRotation);            
        },

        getXSize : function() {
            var chosenDocumentVersion = this.getChosenDocumentVersion();

            var xSizeOne = this.ownDocumentVersion.urx - this.ownDocumentVersion.llx;
            log.info("getXSize says: llx = [" + this.ownDocumentVersion.llx + "], urx = [" + this.ownDocumentVersion.urx + "], "
                + "lly = [" + this.ownDocumentVersion.lly + "], ury = [" + this.ownDocumentVersion.ury + "]");

            var xSizeTwo = 0.0;
            if (chosenDocumentVersion != null) {
                var deltaAngle = this.getDeltaAngle(chosenDocumentVersion);
                var fraction = ((deltaAngle + 360) / 90) % 4;

                log.info("getXSize says about chosen: angle = [" + deltaAngle + "], llx = [" + chosenDocumentVersion.llx + "], urx = [" + chosenDocumentVersion.urx + "], "
                    + "lly = [" + chosenDocumentVersion.lly + "], ury = [" + chosenDocumentVersion.ury + "]");

                if (Math.abs(fraction - 1) < 1e-8 || Math.abs(fraction - 3) < 1e-8) {
                    xSizeTwo = chosenDocumentVersion.ury - chosenDocumentVersion.lly;
                } else {
                    xSizeTwo = chosenDocumentVersion.urx - chosenDocumentVersion.llx;
                }
            }

            return Math.max(xSizeOne, xSizeTwo);            
        },        

        getYSize : function() {
            var chosenDocumentVersion = this.getChosenDocumentVersion();

            var ySizeOne = this.ownDocumentVersion.ury - this.ownDocumentVersion.lly;

            var ySizeTwo = 0.0;
            if (chosenDocumentVersion != null) {
                var deltaAngle = this.getDeltaAngle(chosenDocumentVersion);
                var fraction = ((deltaAngle + 360) / 90) % 4;
                if (Math.abs(fraction - 1) < 1e-8 || Math.abs(fraction - 3) < 1e-8) {
                    ySizeTwo = chosenDocumentVersion.urx - chosenDocumentVersion.llx;
                } else {
                    ySizeTwo = chosenDocumentVersion.ury - chosenDocumentVersion.lly;
                }
            }

            return Math.max(ySizeOne, ySizeTwo);
        },

        resize : function(newSize) {
            var totalHeight = newSize.h;

            var contentWidth = this.captionBarDiv.offsetWidth;
            var contentHeight = totalHeight
                - this.captionBarDiv.offsetHeight
                - 16;

            this.canvasWidget.resize({ w : contentWidth, h : contentHeight });
            this.resizeOverviewWidget();
        },

        resizeOverviewWidget : function() {
            var desiredOverviewWidth = this.overviewWidget.getDesiredWidth();
            var desiredOverviewHeight = this.overviewWidget.getDesiredHeight();            
            var overviewWidth = Math.min(this.canvasWidget.getWidth() - 10, desiredOverviewWidth);
            var overviewHeight = Math.min(this.canvasWidget.getHeight() - 10, desiredOverviewHeight);
            this.overviewWidget.resize({ w : overviewWidth, h : overviewHeight });

            // Function is called during page construction, before initialize was called.  In that situation,
            // the tiledImage doesn't yet exist, and thus cannot be updated.            
            if (this.tiledImage != null) {
                this.tiledImage.updateOverviewImagePosition();
            }
        },

        setData : function() {

        },

        reload : function() {
        },

        updateWidgetState : function() {

        },

        destroy : function() {
            this.inherited(arguments);
        }
    });

    DocumentComparePage.AsyncOperation = {
        LOAD_VERSIONS : "LoadVersions"
    };

    DocumentComparePage.DEFAULT_BASE_SIZE = 640;
    // Default value according to CmpDVTiledPNGImage    
    DocumentComparePage.DEFAULT_RESOLUTION = 72.0;
    DocumentComparePage.DEFAULT_THUMB_SIZE = 320;

    return DocumentComparePage;
});
